/**
  ******************************************************************************
  * @file    stm32_audio_correction.h
  * @author  MCD Application Team
  * @version V1.0.0
  * @date    31-July-2012
  * @brief   This file contains all the functions prototypes for the audio correction.
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT 2012 STMicroelectronics</center></h2>
  *
  * Licensed under MCD-ST Liberty SW License Agreement V2, (the "License");
  * You may not use this file except in compliance with the License.
  * You may obtain a copy of the License at:
  *
  *        http://www.st.com/software_license_agreement_liberty_v2
  *
  * Unless required by applicable law or agreed to in writing, software 
  * distributed under the License is distributed on an "AS IS" BASIS, 
  * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  ******************************************************************************
  */

#ifndef __STM32_AUDIO_CORRECTION_H
#define __STM32_AUDIO_CORRECTION_H

/* Includes ------------------------------------------------------------------*/
#include "audio_app_conf.h"
#include "stm32_audio.h"

#ifdef STM32F2XX
 #include "stm32f2xx.h"
#elif defined(STM32F4XX)
 #include "stm32f4xx.h"
#elif defined(STM32F10X_CL)
 #include "stm32f10x.h"
#endif /* STM32F2XX */

/** @addtogroup STM32_Audio_Utilities
  * @{
  */
  
/** @defgroup AUDIO_CORRECTION
  * @brief This file is the Header file for stm32_audio_correction.c
  * @{
  */ 

/** @defgroup AUDIO_CORRECTION_Exported_Defines
  * @{
  */ 
/* ----------------------------------------------------------------------------

   All the commented configuration parameters are gathered into audio_app_conf.h 
   file. If you already set them in audio_app_conf.h file, there is no need 
   to reconfigure them here.

-----------------------------------------------------------------------------*/

/*-------------------- Optional adjustment Parameters -----------------------*/
/* Uncomment this define to enable state monitoring through LEDs 3 and 4 on STM322xG-EVAL board.
   LED3 toggles each time an UP adjustment is performed (speed up output rate).
   LED4 toggles each time a DOWN adjustment is performed (slow down output rate). 
   This define is applicable for all correction methods and also when no correction method
   is selected in order to show the trend of clock drift. */
#define AUDIO_CORR_USE_LEDS_3_4

#define DIRECTION_IN                              1
#define DIRECTION_OUT                             0

/* Hardware configuration */
#define AUDIO_CORR_TIM                            TIM4
#define AUDIO_CORR_TIM_CLK                        RCC_APB1Periph_TIM4
#define AUDIO_CORR_TIM_IRQn                       TIM4_IRQn
#define AUDIO_CORR_TIM_IRQHANDLER                 TIM4_IRQHandler

/* Preemption and subpriority levels for the audio correction periodic 
  interrupt (generated by timer) */
/* #define AUDIO_CORR_TIM_PRE_PRIO                   3 */
/* #define AUDIO_CORR_TIM_SUB_PRIO                   1 */
/*----------------------------------------------------------------------------*/
/**
  * @}
  */ 

/** @defgroup AUDIO_CORRECTION_Exported_TypesDefinitions
  * @{
  */
typedef enum
{
  SYNCH_NONE     = 0x00, /* OUT/IN: No Synchronization */
 #if (AUDIO_SYNCHRO_CLKSYNTH_ENABLED == 1)
  SYNCH_EXTCLK,          /* OUT: Extrenal Clock Synthesizer synchroization */
 #endif
 #if (AUDIO_SYNCHRO_FEEDBACK_ENABLED == 1)
  SYNCH_FEEDBACK,       /* OUT: Feedback isoc pipe synchronization  */
 #endif
 #if (AUDIO_SYNCHRO_ADDREM_ENABLED == 1)
  SYNCH_ADDREM,         /* OUT: Add/Remove samples synchronization */
 #endif
 #ifdef AUDIO_SYNCHRO_SRC_ENABLED
  SYNCH_SRC,            /* OUT: Source Rate Converter synchronization*/
 #endif
  SYNCH_CLKUPDT  = 0x81  /* IN: Timer clock update synchronization */
}SynchMethod_TypeDef;

typedef struct
{
  SynchMethod_TypeDef  SynchMethod;
  uint32_t             SynchPeriod;
  uint32_t             SynchStep;
  uint32_t             SynchThreshold;
  uint8_t*             pSynchOpt;
  uint32_t             (*Init)     (uint32_t p1, uint32_t p2, uint32_t p3, uint32_t p4);
  uint32_t             (*Process)  (uint32_t* pValue, uint32_t* pDirection, uint32_t* pBuf);
  uint32_t             (*Update)   (uint32_t p1, uint32_t p2, uint32_t p3, uint32_t p4);
  uint32_t             (*DeInit)   (void);
}SynchStruct_TypeDef;
/**
  * @}
  */ 

/** @defgroup AUDIO_CORRECTION_Exported_Macros
  * @{
  */ 
#define ABS_DIFF(a,b)                   (b<a)? (a-b):(b-a)

#ifndef NULL
 #define NULL    0
#endif
/**
  * @}
  */ 
/** @defgroup AUDIO_CORRECTION_Exported_Variables
  * @{
  */ 
/**
  * @}
  */ 

/** @defgroup AUDIO_CORRECTION_Exported_Functions
  * @{
  */
SynchStruct_TypeDef* AUDIO_CORR_SetActiveMethod(uint32_t Direction, SynchMethod_TypeDef Method);
SynchMethod_TypeDef  AUDIO_CORR_GetActiveMethod(uint32_t Direction);

uint32_t AUDIO_CORR_Init(uint32_t AudioFreq);
uint32_t AUDIO_CORR_DeInit(void);
uint32_t AUDIO_CORR_Out_Process(uint32_t* pValue, uint32_t* pDirection, uint32_t* pBuf);
uint32_t AUDIO_CORR_In_Process (uint32_t* pValue, uint32_t* pDirection, uint32_t* pBuf);
uint32_t AUDIO_CORR_Out_Update (uint32_t Opt0, uint32_t Opt1, uint32_t Opt2, uint32_t Opt3);


/* This function should be implemented by the higher layer to get the current 
   shift values */
uint32_t AUDIO_CORR_Out_GetCurrent(uint32_t* pValue, uint32_t* pDirection, uint32_t** pBuff);
uint32_t AUDIO_CORR_In_GetCurrent (uint32_t* pValue, uint32_t* pDirection, uint32_t** pBuff);
/**
  * @}
  */

#endif /* __STM32_AUDIO_CORRECTION_H */

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
