/******************** (C) COPYRIGHT 2007 STMicroelectronics ********************
* File Name          : main.c
* Author             : MCD Application Team
* Version            : V4.0
* Date               : 10/09/2007
* Description        : Main program body
*********************************************************************************
* THE PRESENT SOFTWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
* WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME.
* AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT,
* INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE
* CONTENT OF SUCH SOFTWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING
* INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
*******************************************************************************/

/* Includes ------------------------------------------------------------------*/
#include "71x_lib.h"
#include <stdio.h>

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define UART0_Rx_Pin (0x0001<<8)  /*  TQFP 64: pin N 63 , TQFP 144 pin N 143 */
#define UART0_Tx_Pin (0x0001<<9)  /*  TQFP 64: pin N 64 , TQFP 144 pin N 144 */

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
u16 ADC_Min = 0;       /*ADC register value at 0V*/
u16 ADC_Max = 0;       /*ADC register value at 2.5V*/
u16 ADC_Unsig_Min = 0; /* ADC 12-bit unsigned value at 0V*/
u16 ADC_Unsig_Max = 0; /* ADC 12-bit unsigned value at 2.5V */
u16 SWEEP;
u32 ADC_dV;
u16 ADC_Reg, ADC_Unsig_Result;
u32 Calib_result;

/* Private function prototypes -----------------------------------------------*/
u16 ADC12_UnsigValue(u16 ADC_Reg);
u16 ADC12_ConversionAverage( ADC12_Channels Channel, u16 N);
void my_uart_config (void);

/* Private functions ---------------------------------------------------------*/

/*******************************************************************************
* Function Name  : main
* Description    : Main program
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
int main (void)
{
 #ifdef DEBUG
  debug();
 #endif /* DEBUG */
 
  /* System clocks configuration ---------------------------------------------*/
  /* select CLK2 as RCCU clock source  MCLk=PCLK1=PCLK2= 8 MHZ*/
  RCCU_RCLKSourceConfig(RCCU_CLOCK2);
   
  /* Enable ADC12 and GPIO0 clocks on APB2 */
  APB_ClockConfig (APB2, ENABLE, ADC12_Periph | GPIO1_Periph | GPIO0_Periph);
  
  /* Enable UART0 clock on APB1 */
  APB_ClockConfig (APB1, ENABLE, UART0_Periph);

  /* GPIO configuration -----------------------------------------------------*/

  /*  Configure the Analog input AN3 as HI_AIN_TRI */
  GPIO_Config (GPIO1, (0x0001 << 3), GPIO_HI_AIN_TRI);

  /* Configure the P1.9 NXT button as input TTL */
  GPIO_Config(GPIO1, (0x0001 << 9), GPIO_IN_TRI_TTL);

  
  /* UART0 configuration for windows hyperterminal results display */
  my_uart_config ();

  /* ADC12 configuration -----------------------------------------------------*/

  /*  Initialize the conveter register. */
  ADC12_Init();

  /* Configure the sampling frequency = 500Hz */
  ADC12_PrescalerConfig(500);

  /*  Select the single conversion mode */
  ADC12_ModeConfig (ADC12_SINGLE);

 /* Select the Channel 3 to be converted */
  ADC12_ChannelSelect(ADC12_CHANNEL3);

  /*  Start the Converter */
  ADC12_ConversionStart();
  
  /*Apply 0V on the ADC channel 3  and press the NXT push button */
  printf("Apply 0V on ADC channel 3 and press the NXT push button\r\n");
 
  while(GPIO_BitRead(GPIO1, 9))
  {
  }

  /* Get the ADC conversion result for 0V voltage */
  ADC_Min =  ADC12_ConversionAverage( ADC12_CHANNEL3, 1000);

  /* switch off led on Port P0.0 */
  GPIO_BitWrite(GPIO0, 0, 0);
  printf("ADC conversion result for 0V is 0x%X\r\n", ADC_Min);

  printf("Apply 2.5V on ADC channel 3 and press the NXT push button\r\n");
  
  /* Apply 2.5V on the ADC channel 3 and press the NXT push button */
  while(GPIO_BitRead(GPIO1, 9))
  {
  }
  
  ADC_Max =  ADC12_ConversionAverage( ADC12_CHANNEL3, 1000);

  /* switch on led on Port P0.1 */
  GPIO_BitWrite(GPIO0, 1, 0);

  printf("ADC conversion result for 2.5V is 0x%X\r\n", ADC_Max);

  /* Convert the ADC_Min into 12-bit unsigned value */
  ADC_Unsig_Min = ADC12_UnsigValue(ADC_Min);
  printf("The 12-bit unsigned conversion result of 0V is 0x%X\r\n", ADC_Unsig_Min);

  /*Convert the ADC_Max into 12-bit unsigned value */
  ADC_Unsig_Max = ADC12_UnsigValue(ADC_Max);
  printf("The 12-bit unsigned conversion result of 2.5V is 0x%X\r\n", ADC_Unsig_Max);

  /* Compute the ADC Gain  */
  SWEEP = ADC_Unsig_Max - ADC_Unsig_Min;

  /* Compute the ADC quantium in V */
  if (SWEEP != 0)
  {
    ADC_dV=(2500 * 1000) / SWEEP;
  }
  printf("The ADC computed quatium in microV is %d\r\n", ADC_dV);

  
    /* Wait for the push button to be pressed */
    while(GPIO_BitRead(GPIO1, 9))
    {
    }

   /*  Wait until the availabiliy of data in the specific flags */
    while (ADC12_FlagStatus (ADC12_DA3) == RESET)
    {
    }

    /*  Get the conversion result from the correponding Data register */
    ADC_Reg = ADC12_ConversionValue(ADC12_CHANNEL3);

    /* Convert the ADC Conversion result to 12 bit unsigned value */
    ADC_Unsig_Result = ADC12_UnsigValue(ADC_Reg);

    /* Calibrate the conversion result */
    Calib_result = (ADC_Unsig_Result - ADC_Unsig_Min) * ADC_dV;

    /*  Display the conversion result after calibration */
    printf("Calib_result = %d microV \r\n", Calib_result);
    
    /* Infinite loop */
    while(1)
    {
    }

}

/*******************************************************************************
* Function Name  : ADC12_Unsig_Value
* Description    : This is used for ADC12 result calibration
* Input          : Conv_Res : Conversion result  12 bit two complement
* Output         : None
* Return         : Conversion result calibrated 12 bit unsigned value.
*******************************************************************************/
u16 ADC12_UnsigValue(u16 Conv_Res)
{
  if ((Conv_Res & 0x800) == 0x800)
  {
    return (0x800 - (((~Conv_Res) & 0xFFF) + 1));
  }
  else
  {
    return (Conv_Res + 0x800);
  }
}

/*******************************************************************************
* Function Name  : ADC12_Conversion_Average
* Description    : This function is used to compute the average of an ADC 
*                  channel conversion    
* Input1         : ADC12_Channel: channel selected to be converted
* Input2         : u16 : number of time the channel is converted
* Output         : None
* Return         : Conversion average result 
*******************************************************************************/
u16 ADC12_ConversionAverage(ADC12_Channels Channel, u16 N)
{
  u16 i = 0;
  ADC12_Flags ChannelFlag;
  u32 Conv_Sum = 0;
  u16 Conv_Av =0;

  ChannelFlag = (ADC12_Flags)(1 << (Channel >> 4)) ;

  for(i = 0; i<N; i++)
  {
    /* Wait until the availabilty of converted data */
    while(ADC12_FlagStatus( ChannelFlag) == RESET);

    /* Get the conversion result and add it to Conv_Sum*/
    Conv_Sum += ADC12_ConversionValue(Channel);
   }
  
  Conv_Av = Conv_Sum / N;
 
  return(Conv_Av);
}

/*******************************************************************************
* Function Name  : my_uart_config
* Description    : Configure the UART0 interface
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/ 
void my_uart_config (void)
{
 /*  Configure the GPIO pins */
  GPIO_Config(GPIO0, UART0_Tx_Pin, GPIO_AF_PP);
  GPIO_Config(GPIO0, UART0_Rx_Pin, GPIO_IN_TRI_CMOS);

  /* UART peripheral configuration -------------------------------------------*/

  /*  Configure the UART X */
  UART_OnOffConfig(UART0, ENABLE);       /*  Turn UARTX on */
  UART_FifoConfig (UART0, DISABLE);      /*  Disable FIFOs */
  UART_FifoReset  (UART0 , UART_RxFIFO); /*  Reset the UART_RxFIFO */
  UART_FifoReset  (UART0 , UART_TxFIFO); /*  Reset the UART_TxFIFO */
  UART_LoopBackConfig(UART0 , DISABLE);  /*  Disable Loop Back */
                                         /* Configure the UARTX as following:
                                            - Baudrate = 9600 Bps
                                            - No parity
                                            - 8 data bits
                                            - 1 stop bit */
  UART_Config(UART0,9600,UART_NO_PARITY,UART_1_StopBits,UARTM_8D);
  UART_RxConfig(UART0 ,ENABLE);          /*  Enable Rx */

}

/*******************************************************************************
* Function Name  : sendchar
* Description    : write a character to UART.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void sendchar( char *ch )
{
  UART_ByteSend(UART0,(u8 *)ch);
   if (*ch == 'n')                  
      *ch = '\r';
}

/*******************************************************************************
* Function Name  : fputc
* Description    : write a character to the UART.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
int fputc(int ch, FILE *f)
{
    /* Place your implementation of fputc here */
    /* e.g. write a character to UART */

    char tempch = ch;
#ifdef USE_SERIAL_PORT
    sendchar( &tempch );
#endif
    return ch;
}

/******************* (C) COPYRIGHT 2007 STMicroelectronics *****END OF FILE****/
