/******************** (C) COPYRIGHT 2006 STMicroelectronics ********************
* File Name          : 75x_it.c
* Author             : MCD Application Team
* Date First Issued  : 03/10/2006
* Description        : Main Interrupt Service Routines.
*                      This file can be used to describe all the exceptions
*                      subroutines that may occur within user application.
*                      When an interrupt happens, the software will branch
*                      automatically to the corresponding routine according
*                      to the interrupt vector loaded in the PC register.
*                      The following routines are all empty, user can write code
*                      for exceptions handlers and peripherals IRQ interrupts.
********************************************************************************
* History:
* 07/17/2006 : V1.0
* 03/10/2006 : V0.1
********************************************************************************
* THE PRESENT SOFTWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
* WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME.
* AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT,
* INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE
* CONTENT OF SUCH SOFTWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING
* INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
*******************************************************************************/

/* Includes ------------------------------------------------------------------*/
#include "main.h"

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/

  vu32 SMIReadAddr = 0;          /* SMI read address global address */
  vu32 SMIWriteAddr = 0;         /* SMI write address global address */
  u32 WaveDataLength = 0;        /* Wave player global data length */
  u32 RecordDataLength = 0;      /* Voice recorder global data length */
  vu32 WaveCounter = 0;          /* Wave player and voice recorder global index */

  extern u32 SelectedMaxRecordDataLength;  /* Selected max SMI size dedicated */
                                           /* for voice recording or board test */

  vu32 RecordStatus = 1;  /* Voice recorder global Status */
  vu32 AlarmStatus = 0;   /* Alarm melody global status */
  vu32 KEYSwitch = 1;     /* Detect first or second press on KEY bushbutton */
  u8 VoiceRec_Tab0[256];  /* Table to store first, third,... 256 recorded bytes */
  u8 VoiceRec_Tab1[256];  /* Table to store second, fourth... 256 recorded bytes */
  vu16 RecordSample = 0;  /* Recorded data index in VoiceRec_Tab(0-1) */
  u8* Tab;                /* Pointer on VoiceRec_Tab(0-1) */
  u8 TabIndex = 0;       /* Variable to choice between VoiceRec_Tab(0-1) */
  u8 SampleN_1Value = 0x40;  /* ADC (N-1) sample converted data value */

  extern vu32 SELStatus;     /* SEL pushbutton global Status */

  extern vu32 ScreenStatus;  /* Screen saver global Status */
  u8 ScreenSaveTime = 0;     /* Counter of watchdog interrupt for screen saver */
  extern volatile ErrorCode WaveFileStatusAlarm;  /* Alarm wave file load status */
  extern vu32 LedShowStatus; /* Led show global Status */
  vu32 Index = 0;            /* Index for led Show toggling */

  extern vu16 PeriodmeasureDelay;  /* Counter for period mesure values refresh */
  extern vu32 STOPModeStatus;      /* Low Power Stop mode global status */
  extern vu16 PlayIntroDelay;      /* Play Intro delay: 2sec */
  vu32 WaveDataRead = 0;           /* Played wave data length */
  extern vu8  VoiceRecord;         /* Boardself test record or voice recordering select */
  extern void USB_Istr(void);      /* JoyStickHID function export */
  ErrorStatus OSC4MStartUpStatus02;

/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/

/*******************************************************************************
* Function Name  : Undefined_Handler
* Description    : This function handles Undefined instruction exception.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void Undefined_Handler(void)
{
}

/*******************************************************************************
* Function Name  : FIQ_Handler
* Description    : This function handles FIQ exception.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void FIQ_Handler(void)
{
  /* If SEL pushbutton is pressed */
  if(EIC_GetCurrentFIQChannel() == EXTIT_Line0_FIQChannel)
  {
    /* Disable ScreenSaver */
    ScreenSaver_Cmd(DISABLE);
    /* Set SELStatus to execute SelFunc() function */
    SELStatus = 1;
    /* Enable ScreenSaver */
    ScreenSaver_Cmd(ENABLE);
    /* Clear the EXTIT Line0 pending bit: "SEL" */
    EXTIT_ClearITPendingBit(EXTIT_ITLine0);
    /* Clear EXTIT Line0 FIQ pending bit: "SEL" */
    EIC_ClearFIQPendingBit(EXTIT_Line0_FIQChannel);
  }
  /* Watchdog interrupt generated each ~= 0.25sec */
  if(EIC_GetCurrentFIQChannel() == WATCHDOG_FIQChannel)
  {
    /* When no key is pressed during ~= 30sec */
    if(ScreenSaveTime == 120)
    {
      /* Reset ScreenSaveTime counter */
      ScreenSaveTime = 0;
      /* Set ScreenStatus to activate screen saver */
      ScreenStatus = 1;
    }
    /* Increment watchdog counter */
    ScreenSaveTime++;
    /* Clear WDG FIQ pending bit */
    EIC_ClearFIQPendingBit(WATCHDOG_FIQChannel);
    /* Clear WDG end of Count interrupt pending bit */
    WDG_ClearITPendingBit();
  }
}

/*******************************************************************************
* Function Name  : SWI_Handler
* Description    : This function handles SW exception.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void SWI_Handler(void)
{
}

/*******************************************************************************
* Function Name  : Prefetch_Handler
* Description    : This function handles preftetch abort exception.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void Prefetch_Handler(void)
{
}

/*******************************************************************************
* Function Name  : Abort_Handler
* Description    : This function handles data abort exception.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void Abort_Handler(void)
{
}

/*******************************************************************************
* Function Name  : WAKUP_IRQHandler
* Description    : This function handles External line 15(WAKUP) interrupt
*                  request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void WAKUP_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : TIM2_OC2_IRQHandler
* Description    : This function handles TIM2 Output Compare 2 interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void TIM2_OC2_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : TIM2_OC1_IRQHandler
* Description    : This function handles TIM2 Output Compare 1 interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void TIM2_OC1_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : TIM2_IC12_IRQHandler
* Description    : This function handles TIM2 Input Capture 1 & 2 interrupt
*                  request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void TIM2_IC12_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : TIM2_UP_IRQHandler
* Description    : This function handles TIM2 Update interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void TIM2_UP_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : TIM1_OC2_IRQHandler
* Description    : This function handles TIM1 Output Compare 2 interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void TIM1_OC2_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : TIM1_OC1_IRQHandler
* Description    : This function handles TIM1 Output Compare 1 interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void TIM1_OC1_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : TIM1_IC12_IRQHandler
* Description    : This function handles TIM1 Input Capture 1 & 2 interrupt
*                  request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void TIM1_IC12_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : TIM1_UP_IRQHandler
* Description    : This function handles TIM1 Update interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void TIM1_UP_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : TIM0_OC2_IRQHandler
* Description    : This function handles TIM0 Output Compare 2 interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void TIM0_OC2_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : TIM0_OC1_IRQHandler
* Description    : This function handles TIM0 Output Compare 1 interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void TIM0_OC1_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : TIM0_IC12_IRQHandler
* Description    : This function handles TIM0 Input Capture 1 & 2 interrupt
*                  request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void TIM0_IC12_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : TIM0_UP_IRQHandler
* Description    : This function handles TIM0 Update interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void TIM0_UP_IRQHandler(void)
{
  u8 SampleNValue = 0;

  /* Voice Recording phase */
  if(RecordStatus)
  {
    /* Get the ADC conversion value for sample N */
    /* A numeric RIF filter is applayed to the converted value */
    SampleNValue = 0.89 * ((((ADC_GetConversionValue(ADC_CHANNEL15)) - 0x1F0)) + 0x90)+(1 - 0.89) * SampleN_1Value;

    /* If we haven't reach the max allowed recording SMI size */
    if(WaveCounter < SelectedMaxRecordDataLength)
    {
      /* Select the VoiceRec_Tab destination table */
      if(TabIndex == 0)
      {
        Tab = VoiceRec_Tab0;
      }
      else
      {
        Tab = VoiceRec_Tab1;
      }
      /* Store the converted sample N value */
      Tab[RecordSample++] = SampleNValue;
      /* Set actuel N sample value as N-1 sample for next record */
      SampleN_1Value = SampleNValue;
      /* If the VoiceRec_Tab destination table is full */
      if(RecordSample == 256)
      {
        /* Load SMI with the actual VoiceRec_Tab */
        SMI_FLASH_WritebBuffer(Tab, SMIWriteAddr, 256);
        /* Increment SMIWriteAddr by 256*/
        SMIWriteAddr += 256;
        /* Change the VoiceRec_Tab destination table */
        TabIndex ^= 1 ;
        /* Reset RecordSample counter */
        RecordSample = 0;
        /* Disable ScreenSaver */
        ScreenSaver_Cmd(DISABLE);
      }
      /* Increment WaveCounter which count the recorded wave length */
      WaveCounter++;
    }
    /* If, in the record phase, we reach the max allowed recording SMI size */
    else
    {
      /* Store the new RecordDataLength just at the end of recording */
      RecordDataLength = WaveCounter;
      /* Stop recording */
      VoiceRec_Stop();
      /* Reset RecordStatus */
      RecordStatus = 0;
      /* Set KEYSwitch */
      KEYSwitch = 1;
      /* Reset the record wave length counter */
      WaveCounter = 0;
      /* Enable ScreenSaver */
      ScreenSaver_Cmd(ENABLE);
    }
  }
  /* Wave playing phase */
  else
  {
    /* Read an SMI byte and set it as TIM2 Pulse */
    TIM_SetPulse(TIM2, TIM_Channel_2, *(u8 *) SMIReadAddr);
    /* Increment wave player index */
    WaveCounter++;
    /* Increment SMI read address */
    SMIReadAddr++;
    /* If we reach the WaveDataLength of the wave to play */
    if(WaveCounter == WaveDataLength)
    {
      /* Save Played wave data length to be tested in WavePlayer_Stop */
      WaveDataRead = WaveCounter;
      /* Reset wave player index */
      WaveCounter = 0;
      /* If AlarmStatus is TRUE replay Alarm wave, else stop playing */
      if (AlarmStatus == 1)
      {
        /* Set SMI read address to replay alarm */
        SMIReadAddr = AlarmReadAddr;
      }
      else
      {
        /* Stop wave playing */
        WavePlayer_Stop();
      }
   }
  }

  /* Clear TIM0 interrupt */
  TIM_ClearITPendingBit(TIM0, TIM_IT_Update);
}

/*******************************************************************************
* Function Name  : PWM_OC123_IRQHandler
* Description    : This function handles PWM Output Compare 1,2&3 interrupt
*                  request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void PWM_OC123_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : PWM_EM_IRQHandler
* Description    : This function handles PWM Emergency interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void PWM_EM_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : PWM_UP_IRQHandler
* Description    : This function handles PWM Update interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void PWM_UP_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : I2C_IRQHandler
* Description    : This function handles I2C global interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void I2C_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : SSP1_IRQHandler
* Description    : This function handles SSP1 interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void SSP1_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : SSP0_IRQHandler
* Description    : This function handles SSP0 interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void SSP0_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : UART2_IRQHandler
* Description    : This function handles UART2 global interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void UART2_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : UART1_IRQHandler
* Description    : This function handles UART1 global interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void UART1_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : UART0_IRQHandler
* Description    : This function handles UART0 global interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void UART0_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : CAN_IRQHandler
* Description    : This function handles CAN global interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void CAN_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : USBLP_IRQHandler
* Description    : This function handles USB Low Priority event interrupt
*                  request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void USB_LP_IRQHandler(void)
{
  /* Synchronise USB */
  USB_Istr();
}

/*******************************************************************************
* Function Name  : USBHP_IRQHandler
* Description    : This function handles USB High Priority event interrupt
*                  request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void USB_HP_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : ADC_IRQHandler
* Description    : This function handles ADC global interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void ADC_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : DMA_IRQHandler
* Description    : This function handles DMA global interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void DMA_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : EXTIT_IRQHandler
* Description    : This function handles External lines 14 to 1 interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void EXTIT_IRQHandler(void)
{
/*------------------ "UP" pushbutton pressed ------------------*/
  if(EXTIT_GetITStatus(EXTIT_ITLine3) != RESET)
  {
    /* Disable ScreenSaver */
    ScreenSaver_Cmd(DISABLE);
    /* Execute UpDownFunc function */
    UpDownFunc();
    /* Clear the EXTIT Line3 pending bit: "UP" */
    EXTIT_ClearITPendingBit(EXTIT_ITLine3);
    /* Enable ScreenSaver */
    ScreenSaver_Cmd(ENABLE);
  }

/*----------------- "DOWN" pushbutton pressed -----------------*/
  if(EXTIT_GetITStatus(EXTIT_ITLine8) != RESET)
  {
    /* Disable ScreenSaver */
    ScreenSaver_Cmd(DISABLE);
    /* Execute UpDownFunc function */
    UpDownFunc();
    /* Clear the EXTIT Line8 pending bit: "DOWN" */
    EXTIT_ClearITPendingBit(EXTIT_ITLine8);
    /* Enable ScreenSaver */
    ScreenSaver_Cmd(ENABLE);
  }

/*----------------- "LEFT" pushbutton pressed -----------------*/
  if(EXTIT_GetITStatus(EXTIT_ITLine12) != RESET)
  {
    /* Disable ScreenSaver */
    ScreenSaver_Cmd(DISABLE);
    /* Execute LeftFunc function */
    LeftFunc();
    /* Clear the EXTIT Line12 pending bit: "LEFT" */
    EXTIT_ClearITPendingBit(EXTIT_ITLine12);
    /* Enable ScreenSaver */
    ScreenSaver_Cmd(ENABLE);
  }

/*----------------- "RIGHT" pushbutton pressed ----------------*/
  if(EXTIT_GetITStatus(EXTIT_ITLine13) != RESET)
  {
    /* Disable ScreenSaver */
    ScreenSaver_Cmd(DISABLE);
    /* Execute RightFunc function */
    RightFunc();
    /* Clear the EXTIT Line13 pending bit: "RIGHT" */
    EXTIT_ClearITPendingBit(EXTIT_ITLine13);
    /* Enable ScreenSaver */
    ScreenSaver_Cmd(ENABLE);
  }

/*------------------ "KEY" pushbutton pressed -----------------*/
  if (EXTIT_GetITStatus(EXTIT_ITLine7) != RESET)
  {
    /* "KEY" pressed in STOP mode */
    if(STOPModeStatus == 1)
    {
      /* Execute UpDownFunc to display upper menu */
      UpDownFunc();
      /* Reset STOPModeStatus for next entrance */
      STOPModeStatus = 0;
    }
    /* "KEY" pressed when alarm is palying */
    else if(AlarmStatus == 1)
    {
      /* Stop Alarm playing */
      WavePlayerAlarm_Stop();
    }
    /* "KEY" pressed in voice recording */
    else
    {
      /* "KEY" pressed to start voice recording phase */
      if(KEYSwitch)
      {
        /* Reset the record wave length counter */
        WaveCounter = 0;
        /* Start recording */
        VoiceRec_Start();
        /* Switch KEYSwitch value */
        KEYSwitch = 0;
        /* External Interrupt Disable */
        IntExtOnOffConfig(DISABLE);
      }
      /* "KEY" pressed during the voice recording phase */
      else
      {
        /* Store the new RecordDataLength just at the end of recording */
        RecordDataLength = WaveCounter;
        /* Stop recording */
        VoiceRec_Stop();
        /* Reset WaveCounter to be reused for wave recording or playing */
        WaveCounter = 0;
        /* Set KEYSwitch for next entrance */
        KEYSwitch = 1;
        /* Reset RecordStatus (used to hear the board test recorded voice) */
        RecordStatus = 0;
        /* If we are in the end of voice recording */
        if(VoiceRecord != 0)
        {
          /* External Interrupt Enable */
          IntExtOnOffConfig(ENABLE);
        }
       }
    }
    /* Clear the EXTIT Line7 pending bit: "KEY" */
    EXTIT_ClearITPendingBit(EXTIT_ITLine7);
  }
}

/*******************************************************************************
* Function Name  : MRCC_IRQHandler
* Description    : This function handles MRCC interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void MRCC_IRQHandler(void)
{
  u32 Tmp = 0, Tmp2 = 0;
  u16 j = 0;

  /* Disable ScreenSaver */
  ScreenSaver_Cmd(DISABLE);

  /* Counter value after 30sec:necessary time to enter STANDBY mode */
  Tmp = RTC_GetCounter() + 30;
  /* If no clock is detected: OSC4M is invalid */
  if(MRCC_GetITStatus(MRCC_IT_NCKD) != RESET)
  {
    /* Display "No Clock Detected" message */
    LCD_DisplayString(Line1, "No Clock Detected", BlackText);
    /* Display "  STANDBY in    s" message */
    LCD_DisplayString(Line2, "  STANDBY in   s ", BlackText);
    /* Loop until NCKD is reset */
    while(MRCC_GetFlagStatus(MRCC_FLAG_NCKD) != RESET)
    {
      /* Get the remaining time to enter STANDBY mode */
      j= Tmp-RTC_GetCounter();
      /* Display the remaining time to enter STANDBY mode */
      LCD_DisplayChar(Line2, 91, (j/10)+0x30, BlackText);
      LCD_DisplayChar(Line2, 98, (j%10)+0x30, BlackText);
      /* Get the actual RTC counter value */
      Tmp2 = RTC_GetCounter();
      /* Clear the NCKD flag */
      MRCC_ClearFlag(MRCC_FLAG_NCKD);
      /* After 30s we enter in STANDBY mode */
      if(Tmp2 >= Tmp)
      {
        /* Disable NCKD interrupt */
        MRCC_ITConfig(MRCC_IT_NCKD, DISABLE);
        /* Clear the STDB flag */
        MRCC_ClearFlag(MRCC_FLAG_STDB);
        /* Clear the WKP flag */
        MRCC_ClearFlag(MRCC_FLAG_WKP);
        /* Store the RTC counter value: used to detect a missed alarm when returning from STANDBY mode */
        MRCC_WriteBackupRegister(MRCC_BKP1, RTC_GetCounter());
        /* Enter STANDBY mode */
        MRCC_EnterSTANDBYMode();
      }
    }

    /* Wait for OSC4M start-up */
    OSC4MStartUpStatus02 = MRCC_WaitForOSC4MStartUp();

    if(OSC4MStartUpStatus02 == SUCCESS)
    {
      /* Set CKSYS to 60MHz if the clock is detected on OSC4M */
      MRCC_CKSYSConfig(MRCC_CKSYS_OSC4MPLL, MRCC_PLL_Mul_15);
    }

    /* Re-display the actual menu */
    DisplayMenu();
  }
  /* Clear NCKD flag */
  MRCC_ClearITPendingBit(MRCC_IT_NCKD);

  /* Enable ScreenSaver */
  ScreenSaver_Cmd(ENABLE);
}

/*******************************************************************************
* Function Name  : FLASHSMI_IRQHandler
* Description    : This function handles Flash and SMI global interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void FLASHSMI_IRQHandler(void)
{
}

/*******************************************************************************
* Function Name  : RTC_IRQHandler
* Description    : This function handles RTC global interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void RTC_IRQHandler(void)
{
  /* If counter is equal to 86339: one day was elapsed */
  if((RTC_GetCounter()/3600==23)&&(((RTC_GetCounter()%3600)/60)==59)&&
     (((RTC_GetCounter()%3600)%60)==59)) /* 23*3600+59*60+59=86339 */
  {
    /* Wait until last write operation on RTC registers has finished */
    RTC_WaitForLastTask();
    /* Reset counter value */
    RTC_SetCounter(0x0);
    /* Wait until last write operation on RTC registers has finished */
    RTC_WaitForLastTask();

    /* Increment the date */
    Date_Update();
  }

  /* Alarm interrupt generated */
  if((RTC_GetITStatus(RTC_IT_Alarm) != RESET) && (WaveFileStatusAlarm != Unvalid_RIFF_ID))
  {
    /* Start wave Alarm playing */
    WavePlayerAlarm_Start();
  }

  /* Wait until last write operation on RTC registers has finished */
  RTC_WaitForLastTask();
  /* Clear Second and Alarm interrupt pending bit */
  RTC_ClearITPendingBit(RTC_IT_Second | RTC_IT_Alarm);
  /* Wait until last write operation on RTC registers has finished */
  RTC_WaitForLastTask();
}

/*******************************************************************************
* Function Name  : TB_IRQHandler
* Description    : This function handles TB global interrupt request.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void TB_IRQHandler(void)
{
  /* If LedShowStatus is TRUE: enable leds toggling */
  if(LedShowStatus != 0)
  {
    switch(Index)
    {
      /* LD5 turned on, LD2 turned off */
      case 0:
      {
        GPIO_WriteBit(GPIO2, GPIO_Pin_19, Bit_RESET);
        GPIO_WriteBit(GPIO1, GPIO_Pin_1, Bit_SET);
        Index++;
        break;
      }
      /* LD2 turned on, LD3 turned off */
      case 1:
      {
        GPIO_WriteBit(GPIO1, GPIO_Pin_1, Bit_RESET);
        GPIO_WriteBit(GPIO0, GPIO_Pin_16, Bit_SET);
        Index++;
        break;
      }
      /* LD3 turned on, LD4 turned off */
      case 2:
      {

        GPIO_WriteBit(GPIO0, GPIO_Pin_16, Bit_RESET);
        GPIO_WriteBit(GPIO2, GPIO_Pin_18, Bit_SET);
        Index++;
        break;
      }
      /* LD3 turned on, LD4 turned off */
      case 3:
      {
        GPIO_WriteBit(GPIO2, GPIO_Pin_18, Bit_RESET);
        GPIO_WriteBit(GPIO2, GPIO_Pin_19, Bit_SET);
        Index++;
        break;
      }

      default:
        break;
    }
    /* Reset Index to replay leds switch on sequence  */
    if(Index == 4)
    {
      Index = 0;
    }
  }

  /* Check elapsed time for next measured period display */
  if(PeriodmeasureDelay == PeriodSampling)
  {
    PeriodmeasureDelay = 0;
  }
  /* Increment PeriodmeasureDelay counter */
  PeriodmeasureDelay++;
  /* Clear TB update interrupt pending bit */
  TB_ClearITPendingBit(TB_IT_Update);

  /* Decrement PlayIntroDelay counter */
  PlayIntroDelay--;
}

/******************* (C) COPYRIGHT 2006 STMicroelectronics *****END OF FILE****/
