/*** (c) STMicroelectronics ***************************************************
#
# COMPILER : COSMIC / HIWARE
#
# DESCRIPTION : Public macros 
#
#  SetBit, ClrBit  - set/clear a selected bit in a byte variable.
#  ChgBit          - invert the selected bit value (0->1 or 1->0).
#  AffBit          - set with a chosen value a selected bit in a byte variable.
#  MskBit          - select some bits in a byte variable and copy them in
#                    another byte variable.
#  ValBit          - return the logic value of a selected bit in a byte
#                    variable.
#
#  BitSet, BitClr  - set/clear a defined bit in the hardware register map.
#  BitVal          - return the logic value of a defined bit.
#								                                                              
******************************************************************************/

#ifndef LIB_BITS_H
#define LIB_BITS_H

/* PUBLIC DECLARATIONS *******************************************************/

#ifdef COSMIC
#define Byte unsigned char      // unsigned 8 bit type definition
#define Word unsigned int       // unsigned 16 bit type definition
#define Bool unsigned char	// boolean definition  
#define FALSE	0
#define TRUE	1      
#define NULL 	0
#define EnableInterrupts	_asm("rim");
#define DisableInterrupts	_asm("sim");
#endif

#ifdef HIWARE
#include "Hidef.h"
#endif

/*-----------------------------------------------------------------------------
Method : I
Description : Handle the bit from the character variables. 
Comments :    The different parameters of commands are
              - VAR : Name of the character variable where the bit is located.
              - Place : Bit position in the variable (7 6 5 4 3 2 1 0)
              - Value : Can be 0 (reset bit) or not 0 (set bit)
              The "MskBit" command allows to select some bits in a source
              variables and copy it in a destination var (return the value).
              The "ValBit" command returns the value of a bit in a char 
              variable: the bit is reseted if it returns 0 else the bit is set.
              This method generates not an optimised code yet.
-----------------------------------------------------------------------------*/ 
#define SetBit(VAR,Place)         ( VAR |= (1<<Place) )
#define ClrBit(VAR,Place)         ( VAR &= ((1<<Place)^255) )
#define ChgBit(VAR,Place)         ( VAR ^= (1<<Place) )
#define AffBit(VAR,Place,Value)   ((Value) ? \
                                   (VAR |= (1<<Place)) : \
                                   (VAR &= ((1<<Place)^255)))
#define MskBit(Dest,Msk,Src)      ( Dest = (Msk & Src) | ((~Msk) & Dest) )

#define ValBit(VAR,Place)         (VAR & (1<<Place))



/*-----------------------------------------------------------------------------
Method : II
Description : Handle directly the bit. 
Comments :    The idea is to handle directly with the bit name. For that, it is
              necessary to have RAM area Descriptions (example: HW register...)
              and the following command line for each area.
              This method generates the most optimized code.
-----------------------------------------------------------------------------*/

#define AREA 0x00     /* The area of bits begins at address 0x10. */                   

#define BitClr(BIT)  ( *((Byte *) (AREA+BIT/8)) &= (~(1<<(7-BIT%8))) )
#define BitSet(BIT)  ( *((Byte *) (AREA+BIT/8)) |=   (1<<(7-BIT%8))  )
#define BitVal(BIT)  ( *((Byte *) (AREA+BIT/8)) & (1<<(7-BIT%8)) ) 


#endif // LIB_BITS_H

/*** END OF FILE ***/