/*** (c) STMicroelectronics ***************************************************
#
# PROJECT : ST7 USB LOW SPEED LIBRARY
#
# COMPILER : COSMIC / HIWARE
#
# VERSION : 4.21
#
# DESCRIPTION : USB lib core file  
#
# HISTORY : 08/24/00 	-> Added "USBLibStatus &= ~USB_REMOTE_WAKEUP_ENABLE;" to inform app of RWU disable 
#				-> Added "USBLibStatus |= USB_REMOTE_WAKEUP_ENABLE;" to inform app of RWU enable
# V4.10	Functions GetStatus, ClearFeatureStall, SetFeatureStall, SetupStage UsbBusReset modified
#		Function SetInterface created					
******************************************************************************/   
   
#include "USB.h"   
#include "Macro.h"
#include "Lib_Bits.h"
#include "USB_Opts.h"
#include "Descript.h"
#include "USB_Def.h"
#include "USB_Var.h"                                                                                                            
#include "USB_Rc.h"
#include "USB_Lib.h" 

#ifdef USE_DEVICE_FIRMWARE_UPGRADE
#include "DFUCore.h"
#include "DFUDescript.h"
#endif

#ifdef HIWARE
#include MAP_FILE                               
#pragma CODE_SEG USBLIB_ROM
#endif

#ifdef INIT_APP_ON_USB_RESET
extern void User_MCU_Init(void); // Call-back function
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : UsbBusReset
INPUT/OUTPUT : None
DESCRIPTION  : Reset registers and variables associated to USB and Disable com on EP0
				Called when a USB Bus Reset is received from Host.
-----------------------------------------------------------------------------*/
void UsbBusReset (void)
{

#ifdef INIT_APP_ON_USB_RESET
    User_MCU_Init();
#endif

	// USB Variables initialization

	UsbReport[0] = 0;					// Initialize UsbReport variable
	UsbReport[1] = 0;					// Initialize UsbReport variable
	ConfigValue	 = 0;					// Default Configuration.
	UsbCtrFlag			= 0;			// Clear CTR Int. flags
	USBTransferStatus	= 0;			// Clear USB Control status flags.
	UsbInfo	= DEFAULT_STATE;			// Add =0|Config=0|Device awake   
	USBLibStatus 	&= 0x07;			// Reset lib status bits. Keep product selection.

    USBDataXferStatus = FIRST_DATA;           // Used to send/receive more than 8 data
    
#ifdef USE_ENDPOINT0_OUT
	OUT_DataNumberEP0 = 0;				// Reset Number of received data on EP0 when OUT Ctr.
#endif
    
#ifdef USE_ENDPOINT1_OUT
	OUT_DataNumberEP1 = 0;				// Reset Number of received data on EP1 when OUT Ctr.
#endif

#ifdef USE_ENDPOINT2_OUT
	OUT_DataNumberEP2 = 0;				// Reset Number of received data on EP2 when OUT Ctr.
#endif

	DeviceStatusInfo	= 0;			// Remote Wake up and power status flags initialization.

#ifdef USE_DEVICE_FIRMWARE_UPGRADE
	if (USBLibStatus & DFU_MODE)
	{
		if (ValBit(DFU_Config_Desc[7], 6))
			DeviceStatusInfo |= SELF_POW;	// Self Powered Device                                          
	}
	else
#endif
#ifdef USE_PRODUCT2
	if (USBLibStatus & PRODUCT2)
	{
		if (ValBit(Descriptors.P2_Config[7], 6))
			DeviceStatusInfo |= SELF_POW;	// Self Powered Device                                          
	}
	else
#endif
	{
		if (ValBit(Descriptors.P1_Config[7], 6))
			DeviceStatusInfo |= SELF_POW;	// Self Powered Device                                          
	}

	// Endpoint 1 & 2 address initialization
	USBEP1RB =  0x01;				// Set endpoint 1 internal address
	USBEP2RB =  0x02;				// Set endpoint 2 internal address
	// REM: EP1 & EP2 are automatically disabled by hardware when a bus reset occurs.

	// Endpoint0 initialization. First Transaction --> SETUP ONLY EXPECTED ON EP0
	// All other EPs are disable 
	SetEP0TxStatus_IT(STALL);			//  Next IN will be stalled.
	SetEP0RxStatus_IT(STALL);			//  Next OUT will be stalled.

	// Initialize USBIMR
	USBIMR = 0xA6;					// set USB interrupt mask. Enable SUSP|CTR|ESUSP|RESET

}

/*-----------------------------------------------------------------------------
ROUTINE NAME : EnableAppli()
INPUT/OUTPUT : None
DESCRIPTION  : Enable Application EndPoints. Function called when device has been configured
-----------------------------------------------------------------------------*/
void EnableAppli(void)
{
	// Endpoint 1 and/or Endpoint 2 enable
#ifdef USE_ENDPOINT1_IN			// Endpoint 1 used in interrupt input pipe
	SetEP1TxStatus(NAK);		// Set the Endpoint 1 Tx status to NAK -> IN token will be NACKED
	ResetDataToggleEP1_IN;		
	DeviceStatusInfo &= ~ENDPOINT1_STALLED;	// Reset the STALL bit Status Info 
#endif                          

#ifdef USE_ENDPOINT1_OUT		// Endpoint 1 used in interrupt output pipe
	SetEP1RxStatus(VALID);		// Next Data reception on EP1 will be ACK -> OUT token will be ACKED
	ResetDataToggleEP1_OUT;		
	DeviceStatusInfo &= ~ENDPOINT1_STALLED;	// Reset the STALL bit Status Info 
#endif

#ifdef USE_ENDPOINT2_IN			// Endpoint 2 used in interrupt input pipe                                                                                                                                                                                                                                                                                                                              
	SetEP2TxStatus(NAK);		// Set the Endpoint 2 Tx status to NAK-> IN token will be NACKED
	ResetDataToggleEP2_IN;		
	DeviceStatusInfo &= ~ENDPOINT2_STALLED;	// Reset the STALL bit Status Info 
#endif                          

#ifdef USE_ENDPOINT2_OUT		// endpoint 2 used in interrupt output pipe                                             
	SetEP2RxStatus(VALID);		// Next Data reception on EP2 will be ACK -> OUT token will be ACKED                                                
	ResetDataToggleEP2_OUT;		
	DeviceStatusInfo &= ~ENDPOINT2_STALLED;	// Reset the STALL bit Status Info 
#endif 
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : DisableAppli()
INPUT/OUTPUT : None
DESCRIPTION  : Disable Application Endpoints. Function called when device config is set back to 0.
-----------------------------------------------------------------------------*/
void DisableAppli(void)
{
#ifdef USE_ENDPOINT1_IN			// Endpoint 1 used in interrupt input pipe
	SetEP1TxStatus(DISABLE);				// Set the Endpoint 1 Tx status to DISABLE
	DeviceStatusInfo &= ~ENDPOINT1_STALLED;	// Reset the STALL bit Status Info 
#endif                          

#ifdef USE_ENDPOINT1_OUT		// Endpoint 1 used in interrupt output pipe
	SetEP1RxStatus(DISABLE);		// First Data reception will be NOT ANSWERED
	DeviceStatusInfo &= ~ENDPOINT1_STALLED;	// Reset the STALL bit Status Info 
#endif

#ifdef USE_ENDPOINT2_IN			// Endpoint 2 used in interrupt input pipe                                                                                                                                                                                                                                                                                                                              
	SetEP2TxStatus(DISABLE);		// Set the Endpoint 2 Tx status to DISABLE
	DeviceStatusInfo &= ~ENDPOINT2_STALLED;	// Reset the STALL bit Status Info 
#endif                          

#ifdef USE_ENDPOINT2_OUT		// endpoint 2 used in interrupt output pipe                                             
	SetEP2RxStatus(DISABLE);      // First Data reception will be NOT ANSWERED                                              
	DeviceStatusInfo &= ~ENDPOINT2_STALLED;	// Reset the STALL bit Status Info 
#endif 
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetConfiguration
INPUT/OUTPUT : None
DESCRIPTION  : Assign configuration value sent by Host & reset Stalled EPs.
-----------------------------------------------------------------------------*/
void SetConfiguration(void)
{	// Valid request in ADDRESS_STATE and CONFIGURED_STATE

    Bool ok = FALSE;

#ifdef USE_DEVICE_FIRMWARE_UPGRADE
    if (USBLibStatus & DFU_MODE) {
      if ((USBwValue[0] == 0) || (USBwValue[0] == DFU_Config_Desc[5])) ok = TRUE;
    }
    else {
      if ((USBwValue[0] == 0) || (USBwValue[0] == Descriptors.P1_Config[5])) ok = TRUE;
    }
#else
	if ((USBwValue[0] == 0) || (USBwValue[0] == Descriptors.P1_Config[5])) ok = TRUE;
#endif

    if (ok)
	{
		ConfigValue =  USBwValue[0];
		if (ConfigValue != 0)
		{
			EnableAppli();
			UsbInfo = (UsbInfo & ~USB_STATE) | CONFIGURED_STATE;	// Device is now configured
			USBLibStatus |= USB_CONFIGURED;
		}
		else
		{
			DisableAppli();			 
			UsbInfo = (UsbInfo & ~USB_STATE) | ADDRESS_STATE;	// Device has been set back to Address State
			USBLibStatus &= ~USB_CONFIGURED;
		}
	}
	else
		RequestError();				// Inappropriate Request
}

#ifdef USE_USB_GET_INTERFACE
/*-----------------------------------------------------------------------------
ROUTINE NAME : GetInterface
INPUT/OUTPUT : None
DESCRIPTION  : Send the selected alternate setting for the specified interface
               to the Host (single interface -> alternate setting =0)
-----------------------------------------------------------------------------*/
void GetInterface(void)                          
{
  if (!(UsbInfo & CONFIGURED_STATE)) 
  {
    	RequestError();	// Inappropriate Request. device not configured
    	return;
  }

  if((USBbmRequestType & RECIPIENT) == REINTERFACE)
  {
		if (USBwIndex == 0)		// Interface 0
		{
			UsbReport[0] = 0;   // Send a 0 according to spec.
		}
    	else RequestError();	// Inappropriate Request. Non existing interface.
  }
  else RequestError();	// Inappropriate Request. Non existing interface.
}
#endif

#ifdef USE_USB_SET_INTERFACE
/*-----------------------------------------------------------------------------
ROUTINE NAME : SetInterface
INPUT/OUTPUT : None
DESCRIPTION  : Allow the Host to select an alternate setting for the specified interface               
-----------------------------------------------------------------------------*/
void SetInterface(void)
{
  if (!(UsbInfo & CONFIGURED_STATE)) 
  {
    RequestError(); // Inappropriate Request. device not configured
    return;
  }

  if ((USBbmRequestType & RECIPIENT) == REINTERFACE)
  {
    if((USBwIndex == 0) && (USBwValue[0] == 0) && (USBwValue[1] == 0))
    {
      EnableAppli();
    }
    else RequestError(); // Inappropriate Request. Non existing interface.
  }  
  else RequestError(); // Inappropriate Request. Non existing interface.
}
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : GetStatus
INPUT/OUTPUT : None
DESCRIPTION  : Build status report into UsbReport[] global variable.
-----------------------------------------------------------------------------*/
void GetStatus()
{	// REM: Request Valid only on EP0 while in ADDRESS STATE

	switch (USBbmRequestType & RECIPIENT)
	{

	case REINTERFACE :	
          if (UsbInfo & CONFIGURED_STATE) {
            if (USBwIndex == 0) { // Interface 0
              UsbReport[0] = 0; // Send a 0 Word according to spec.
              UsbReport[1] = 0; // Send a 0 Word according to spec.
            }
#if defined(USE_P1_INTERFACE1) || defined(USE_P2_INTERFACE1)
            else if (USBwIndex == 1) { // Interface 1
                   UsbReport[0] = 0; // Send a 0 Word according to spec.
                   UsbReport[1] = 0; // Send a 0 Word according to spec.
            }
#endif
            else { // Non existing interface
              RequestError();
            }
          }
          else { // Request only valid in the Configurated state
            RequestError();
          }
	break;		

	case REDEVICE :
		{
			UsbReport[0] = DeviceStatusInfo & 0x03;
		}
		break; // REM: USBwIndex = 0.

	case REENDPOINT : 
		if (USBwIndex == 0x80)	// EP0 IN
		{
			UsbReport[0] = (DeviceStatusInfo>>2) & 0x01;	// Send Status for EP0 
		}
		else if (USBwIndex == 0x00)	// EP0 OUT
		{
			UsbReport[0] = (DeviceStatusInfo>>2) & 0x01;	// Send Status for EP0 
		}
		else if (!(UsbInfo & CONFIGURED_STATE))	// Valid request on EP0 ONLY, while in ADDRESS STATE.
		{
			RequestError();		// Inappropriate Request. 
		}

		// Device is in CONFIGURED_STATE
#ifdef USE_ENDPOINT1_IN
		else if (USBwIndex == 0x81)
		{
			UsbReport[0] = (DeviceStatusInfo>>4) & 0x01;
		}
#endif
#ifdef USE_ENDPOINT1_OUT
		else if (USBwIndex == 0x01)
		{
			UsbReport[0] = (DeviceStatusInfo>>4) & 0x01;
		}
#endif

#ifdef USE_ENDPOINT2_IN
		else if (USBwIndex == 0x82)
		{
			UsbReport[0] = (DeviceStatusInfo>>5) & 0x01;
		}
#endif
#ifdef USE_ENDPOINT2_OUT
		else if (USBwIndex == 0x02)
		{
			UsbReport[0] = (DeviceStatusInfo>>5) & 0x01;
		}
#endif
		else
		{
			RequestError();		// Inappropriate Request. Endpoint does not exist.
		}
		break;				// End of Recipient = Endpoint

	default : 
		RequestError();		// Inappropriate Request -> Recipient does not exist.
	}						// End of switch Recipient
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : ClearFeatureStall
INPUT/OUTPUT : None
DESCRIPTION  : Clear or disable a specific feature for the specified recipient.
-----------------------------------------------------------------------------*/
void ClearFeatureStall(void)                                           
{	// REM: Request Valid only on EP0 while in ADDRESS STATE

	if ((USBbmRequestType & RECIPIENT) == REDEVICE)
	{
#ifdef USE_USB_REMOTE_WAKE_UP
#ifdef USE_DEVICE_FIRMWARE_UPGRADE
		if (USBLibStatus & DFU_MODE)
		{
			RequestError();		//  Remote WU not supported on the DFU code loader.
		}
		else
#endif
		{
		  if ((USBwValue[0] | USBwValue[1]) == 0x01) { // Check if Feature selector is Remote WakeUp (wValue = 0x0001)
			DeviceStatusInfo &= ~REMOTE_WAKE_UP_ENABLE;	// Disable Remote WU
			USBLibStatus &= ~USB_REMOTE_WAKEUP_ENABLE;	// Inform app that RWU has been disabled
		  }
		  else RequestError(); // Feature not supported on device : STALL it 	
		}
#else
		RequestError();		//  Remote WU not supported by any product
#endif
	}
	else if ((USBbmRequestType & RECIPIENT) == REENDPOINT)
	{       
	        if ((USBwValue[0] | USBwValue[1]) != 0) { // Check if Feature selector is different of ENDPOINT_HALT 
	          RequestError();                         // If it's the case, STALL it (spec 9.4.1)
	          return;
	        }
		if (USBwIndex == 0x80) // EP0 IN
		{
#ifdef USE_EP0_CLEAR_FEATURE
			return; // In any cases, we'll have both EP0 IN and EP0 OUT STALLed after the SETUP
			        // completion, because that's how they should.
#else		
			RequestError();	// Inappropriate Request -> ClearFeature(HALT_ENDPOINT) is OPTIONAL on EP0. 
#endif
		}
		else if (USBwIndex == 0x00) // EP0 OUT
		{   
#ifdef USE_EP0_CLEAR_FEATURE
			return; // In any cases, we'll have both EP0 IN and EP0 OUT STALLed after the SETUP
			        // completion, because that's how they should.
#else		
			RequestError();	// Inappropriate Request -> ClearFeature(HALT_ENDPOINT) is OPTIONAL on EP0. 
#endif
		}
		else if (!(UsbInfo & CONFIGURED_STATE))		// Valid request on EP0 ONLY, while in ADDRESS STATE.
		{
			RequestError();		// Inappropriate Request. 
		}

		// Device is in CONFIGURED_STATE
#ifdef USE_ENDPOINT1_IN
		else if (USBwIndex == 0x81)
		{
			SetEP1TxStatus(NAK);		// Set the Endpoint 1 Tx status to NAK -> IN token will be NACKED
			ResetDataToggleEP1_IN;		
			DeviceStatusInfo &= ~ENDPOINT1_STALLED;	// Reset the STALL bit Status Info 
		}
#endif
#ifdef USE_ENDPOINT1_OUT
		else if (USBwIndex == 0x01)
		{
			SetEP1RxStatus(VALID);		// Next Data reception on EP1 will be ACK -> OUT token will be ACKED
			ResetDataToggleEP1_OUT;		
			DeviceStatusInfo &= ~ENDPOINT1_STALLED;	// Reset the STALL bit Status Info 
		}
#endif
#ifdef USE_ENDPOINT2_IN
		else if (USBwIndex == 0x82)
		{
			SetEP2TxStatus(NAK);		// Set the Endpoint 2 Tx status to NAK-> IN token will be NACKED
			ResetDataToggleEP2_IN;		
			DeviceStatusInfo &= ~ENDPOINT2_STALLED;	// Reset the STALL bit Status Info 
		}
#endif
#ifdef USE_ENDPOINT2_OUT
		else if (USBwIndex == 0x02)
		{
			SetEP2RxStatus(VALID);		// Next Data reception on EP2 will be ACK -> OUT token will be ACKED                                                
			ResetDataToggleEP2_OUT;		
			DeviceStatusInfo &= ~ENDPOINT2_STALLED;	// Reset the STALL bit Status Info 
		}
#endif
		else
			RequestError();		// Inappropriate Request. Endpoint does not exist.
	}	// End of Recipient = Endpoint.					
	else
		RequestError();		// Inappropriate Request. Recipient does not exist.
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetFeatureStall
INPUT/OUTPUT : None
DESCRIPTION  : Set or enable a specific feature for the specified recipient.
-----------------------------------------------------------------------------*/
void SetFeatureStall(void)
{	// REM: Request Valid only on EP0 while in ADDRESS STATE
	if ((USBbmRequestType & RECIPIENT) == REDEVICE)
	{
#ifdef USE_USB_REMOTE_WAKE_UP
#ifdef USE_DEVICE_FIRMWARE_UPGRADE
		if (USBLibStatus & DFU_MODE)
		{
			RequestError();		//  Remote WU not supported on DFU code loader.

		}
		else
#endif
		{
		  if ((USBwValue[0] | USBwValue[1]) == 0x01) { // Check if Feature selector is Remote WakeUp (wValue = 0x0001)
			DeviceStatusInfo |= REMOTE_WAKE_UP_ENABLE;	// Enable Remote WU
			USBLibStatus |= USB_REMOTE_WAKEUP_ENABLE;	// Inform app that RWU has been enabled
		  }
		  else RequestError(); // Feature not supported on device : STALL it	
		}
#else
		RequestError();		//  Remote WU not supported by any product
#endif
	}
	else if ((USBbmRequestType & RECIPIENT) == REENDPOINT)
	{       
	        if ((USBwValue[0] | USBwValue[1]) != 0) { // Check if Feature selector is different of ENDPOINT_HALT 
	          RequestError();                         // If it's the case, STALL it (spec 9.4.1)
	          return;
	        }
		if (USBwIndex == 0x80) // EP0 IN
		{
			RequestError();	// Inappropriate Request -> SetFeature(HALT_ENDPOINT) is OPTIONAL on EP0. 
		}
		else if (USBwIndex == 0x00) // EP0 OUT
		{
			RequestError();	// Inappropriate Request -> SetFeature(HALT_ENDPOINT) is OPTIONAL on EP0. 
		}
		else if (!(UsbInfo & CONFIGURED_STATE))	// Valid request on EP0 ONLY, while in ADDRESS STATE.
		{
			RequestError();		// Inappropriate Request. 
		}

		// Device is in CONFIGURED_STATE
#ifdef USE_ENDPOINT1_IN
		else if (USBwIndex == 0x81)
		{
			DeviceStatusInfo |= ENDPOINT1_STALLED;	// Set the STALL bit Status Info 
			SetEP1TxStatus(STALL);	// Stall IN Requests
		}
#endif
#ifdef USE_ENDPOINT1_OUT
		else if (USBwIndex == 0x01)
		{
			DeviceStatusInfo |= ENDPOINT1_STALLED;	// Set the STALL bit Status Info 
			SetEP1RxStatus(STALL);	// Stall OUT Requests
		}
#endif
#ifdef USE_ENDPOINT2_IN
		else if (USBwIndex == 0x82)
		{
			DeviceStatusInfo |= ENDPOINT2_STALLED;	// Set the STALL bit Status Info 
			SetEP2TxStatus(STALL);	// Stall IN Requests
		}
#endif
#ifdef USE_ENDPOINT2_OUT
		else if (USBwIndex == 0x02)
		{
			DeviceStatusInfo |= ENDPOINT2_STALLED;	// Set the STALL bit Status Info 
			SetEP2RxStatus(STALL);	// Stall OUT Requests
		}
#endif
		else
			RequestError();		// Inappropriate Request. Endpoint does not exist.
	}	// End of Recipient = Endpoint.					
	else
		RequestError();		// Inappropriate Request. Recipient does not exist.
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetupStage
INPUT/OUTPUT : None
DESCRIPTION  : Process the Setup Stage token in Control Transfer.
-----------------------------------------------------------------------------*/
void SetupStage(void)
{

	Word ActualDescLength,wLength_temp;						// Use to store the initial String Desc length 				

	USBbmRequestType = EP0OutBuffer[0];							// save current USBbmRequestType
	USBbRequest      = EP0OutBuffer[1];							// save current USBbRequest
	USBwValue[0]	      = EP0OutBuffer[2];							// save low byte of current wValue 
	USBwValue[1]	      = EP0OutBuffer[3];							// save high byte of current wValue 
	USBwIndex		  = MAKEWORD(EP0OutBuffer[5],EP0OutBuffer[4]);	// save current USBwIndex
    USBwLength		  = MAKEWORD(EP0OutBuffer[7],EP0OutBuffer[6]);	// save current USBwLength
/* REM:	USBwValue[0], USBwValue[1], USBwIndex, USBwLength may be modified to keep track of the on-going transfer.
		Thus, USBwIndex is the current Descriptor Index, USBwLength is the remaining number of bytes to be sent.
		USBwLength stands also for the remaining number of bytes to receive from Host, during OUT Data Stage. 
-->>	This avoid duplicating global variables */

      USBDataXferStatus = FIRST_DATA; // Used to send/receive more than 8 data

	if (USBwLength == 0) // wLength equals to 0 means there is no Data stage.		
	{	// Requests without data stages are processed here.              
	
		USBTransferStatus = NO_DATA_STAGE;	// A Status In will follow the Setup Stage

		switch (USBbmRequestType & TYPE)	
		{
		case STANDARD :
			switch (USBbRequest)
			{
			case CLEAR_FEATURE_STALL:		ClearFeatureStall(); 
								return;
			case SET_FEATURE_STALL:			SetFeatureStall(); 
											return;
			case SET_ADDRESS:				if (USBwValue[0] > 0x7F)
									{	// USB address is coded on 7 bits
										RequestError();		// Inappropriate Request
									}
									else
									{
										USBTransferStatus |= ADDRESS2SET; 
									}
											return; // Address actually assigned at completion of Status Stage
			case SET_CONFIGURATION:				SetConfiguration(); 
											return;
#ifdef USE_USB_SET_INTERFACE
			case SET_INTERFACE:				SetInterface();
											return;
#endif											
			default:					RequestError();		// Inappropriate Request
											return;
			}

		case CLASS :
			switch (USBbRequest)
			{
#ifdef USE_HID_IDLE
			case SET_IDLE :					USBLibStatus |= APP_REQUEST;			
											return;
#endif
#ifdef USE_HID_PROTOCOL
			case SET_PROTOCOL :				USBLibStatus |= APP_REQUEST;
											return;
#endif      
#ifdef USE_DEVICE_FIRMWARE_UPGRADE
			case DFU_DETACH :
			case DFU_CLRSTATUS :
			case DFU_ABORT :
			case DFU_DNLOAD :
											USBLibStatus |= APP_REQUEST;
											return;

#endif
			default :						RequestError();		// Inappropriate Request
											return;
			}
#ifdef USE_VENDOR_REQUESTS
		case VENDOR :						USBLibStatus |= APP_REQUEST;
											return;
#endif
		default:							RequestError();		// Inappropriate Request
											return;
		}		// End of Request Type selection
	}		// End of NO Data Stage Section 

	else
	{// Following Requests require a Data Stage
		if (USBbmRequestType & EP_IN)
		{// a 'data stage in' will follow the 'Setup stage'
			USBTransferStatus = DATA_STAGE_IN;	// A Data Stage IN will follow the Setup Stage
												// -> We need to select and prepare Data to Send
			switch (USBbmRequestType & TYPE)	
			{
			case STANDARD:
				switch (USBbRequest)
				{
				case GET_DESCRIPTOR :
					switch (USBwValue[1])								// Descriptor Type Selection
					{

					case DEVICE_DESC :	
#ifdef USE_DEVICE_FIRMWARE_UPGRADE
						if (USBLibStatus & DFU_MODE)
						{
							DescAddBasePointer = DFU_Device_Desc;	// Device Desc Base Pointer
                            ActualDescLength = DFU_Device_Desc[0]; // Device Desc Length
						}
						else
#endif
#ifdef USE_PRODUCT2
						if (USBLibStatus & PRODUCT2)
						{
							DescAddBasePointer = Descriptors.P2_Device;	// Device Desc Base Pointer
                            ActualDescLength = Descriptors.P2_Device[0]; // Device Desc Length
						}
						else
#endif
						{
							DescAddBasePointer = Descriptors.P1_Device;	// Device Desc Base Pointer
                            ActualDescLength = Descriptors.P1_Device[0]; // Device Desc Length
						}						
						break;	

					case STRING_DESC : 
						switch (USBwValue[0])							// String Descriptor Offset 
						{

						case 0 :				
#ifdef USE_DEVICE_FIRMWARE_UPGRADE
						    if (USBLibStatus & DFU_MODE)
						    {
                                DescAddBasePointer = DFU_Lang_String;	// String Desc Base Pointer
								ActualDescLength = DFU_Lang_String[0];
                            }
                            else
#endif
							{
								DescAddBasePointer = Descriptors.Lang;	// String Desc Base Pointer
								ActualDescLength = Descriptors.Lang[0];
							}
							break;

						case INDEX_MANUFACTURER :
#ifdef USE_DEVICE_FIRMWARE_UPGRADE
						    if (USBLibStatus & DFU_MODE)
						    {
                                DescAddBasePointer = DFU_Manuf_String;	// String Desc Base Pointer
								ActualDescLength = DFU_Manuf_String[0];
                            }
                            else
#endif				
							{
								DescAddBasePointer = Descriptors.Manuf;	// String Desc Base Pointer
								ActualDescLength = Descriptors.Manuf[0]; 
							}
							break;

						case INDEX_NAME :				
#ifdef USE_DEVICE_FIRMWARE_UPGRADE
							if (USBLibStatus & DFU_MODE)
							{
								DescAddBasePointer = DFU_Name_String;	// String Desc Base Pointer
								ActualDescLength = DFU_Name_String[0]; 
							}
							else
#endif
#ifdef USE_PRODUCT2
							if (USBLibStatus & PRODUCT2)
							{
								DescAddBasePointer = Descriptors.P2_Name;	// String Desc Base Pointer
								ActualDescLength = Descriptors.P2_Name[0]; 
							}
							else
#endif
							{
								DescAddBasePointer = Descriptors.P1_Name;	// String Desc Base Pointer
								ActualDescLength = Descriptors.P1_Name[0]; 
							}
							break;

#ifdef USE_SERIAL_NUMBER
						case INDEX_SERIALNUM :				
/*
#ifdef USE_DEVICE_FIRMWARE_UPGRADE
							if (USBLibStatus & DFU_MODE)
							{
								DescAddBasePointer = DFU_Serial_String;	// String Desc Base Pointer
								ActualDescLength = DFU_Serial_String[0]; 
							}
							else
#endif
*/
#ifdef USE_PRODUCT2
							if (USBLibStatus & PRODUCT2)
							{
								DescAddBasePointer = Descriptors.P2_Serial;	// String Desc Base Pointer
								ActualDescLength = Descriptors.P2_Serial[0]; 
							}
							else
#endif
							{
                  				DescAddBasePointer = Descriptors.P1_Serial;	// String Desc Base Pointer
								ActualDescLength = Descriptors.P1_Serial[0]; 
							}
							break;
#endif

#ifdef USE_ADD_STRINGS
						case INDEX_ADD_STRING :				
#ifdef USE_DEVICE_FIRMWARE_UPGRADE
							if (USBLibStatus & DFU_MODE)
							{
								DescAddBasePointer = DFU_Add_String;	// String Desc Base Pointer
								ActualDescLength = DFU_Add_String[0]; 
							}
							else
#endif
							{
								DescAddBasePointer = Descriptors.Add;	// String Desc Base Pointer
								ActualDescLength = Descriptors.Add[0]; 
							}
							break;
#endif
						default :
							RequestError();		// Inappropriate Request
							return;
						}				// End of switch(USBwValue[0])
						USBwIndex = 0;	// To start reading at the begining of the Descriptor.
						break;			// End of String Desc section.

					case CONFIGURATION_DESC :	
#ifdef USE_DEVICE_FIRMWARE_UPGRADE
						if (USBLibStatus & DFU_MODE)
						{
							DescAddBasePointer = DFU_Config_Desc;	// Configuration Desc Base Pointer
							ActualDescLength = MAKEWORD(DFU_Config_Desc[3],DFU_Config_Desc[2]); 
						}
						else
#endif
#ifdef USE_PRODUCT2
						if (USBLibStatus & PRODUCT2)
						{
							DescAddBasePointer = Descriptors.P2_Config;	// Configuration Desc Base Pointer
							ActualDescLength = MAKEWORD(Descriptors.P2_Config[3],Descriptors.P2_Config[2]); 
						}
						else
#endif
						{
							DescAddBasePointer = Descriptors.P1_Config;	// Configuration Desc Base Pointer
							ActualDescLength = MAKEWORD(Descriptors.P1_Config[3],Descriptors.P1_Config[2]); 
						}

						break;

#ifdef HID_APPLICATION
					case HID_DESC :	

#ifdef USE_PRODUCT2
						if (USBLibStatus & PRODUCT2)
						{
							DescAddBasePointer = Descriptors.P2_Config;	// HID Desc Base Pointer(= Configuration)
							ActualDescLength = Descriptors.P2_Config[0]; 
							if (USBwIndex == 0)		// Select Interface
								USBwIndex = 18;					// Initial HID Desc Pointer interface 0
#ifdef USE_P2_INTERFACE1
							else if (USBwIndex == 1)
								USBwIndex = 43;					// Initial HID Desc Pointer interface 1
#endif
							else
							{
								RequestError();		// Inappropriate Request -> Interface number not valid.
								return;
							}
						}
						else
#endif
						{
							DescAddBasePointer = Descriptors.P1_Config;	// HID Desc Base Pointer(= Configuration)
							ActualDescLength = Descriptors.P1_Config[0]; 
							if (USBwIndex == 0)		// Select Interface
								USBwIndex = 18;					// Initial HID Desc Pointer interface 0
#ifdef USE_P1_INTERFACE1
							else if (USBwIndex == 1)
								USBwIndex = 43;					// Initial HID Desc Pointer interface 1
#endif
							else
							{
								RequestError();		// Inappropriate Request -> Interface number not valid.
								return;
							}
						}
						break;

					case REPORT_DESC :	
#ifdef USE_PRODUCT2
						if (USBLibStatus & PRODUCT2)
						{
							if (USBwIndex == 0)	// Select Interface
							{
								DescAddBasePointer = Descriptors.P2_Report;// Report Desc Base Pointer
								ActualDescLength = P2_REPTDESC_LEN;
							}
#ifdef USE_P2_INTERFACE1
							else if (USBwIndex == 1)
							{
								USBwIndex = 0;	// to start at begining of desc
								DescAddBasePointer = Descriptors.P2_I1_Report;// Report Desc Base Pointer
								ActualDescLength = P2_I1_REPTDESC_LEN;
							}
#endif
							else
							{
								RequestError();		// Inappropriate Request -> Interface number not valid.
								return;
							}
						}
						else
#endif
						{
							if (USBwIndex == 0)	// Select Interface
							{
								DescAddBasePointer = Descriptors.P1_Report;// Report Desc Base Pointer
								ActualDescLength = P1_REPTDESC_LEN;
							}
#ifdef USE_P1_INTERFACE1
							else if (USBwIndex == 1)
							{
								USBwIndex = 0;	// to start at begining of desc
								DescAddBasePointer = Descriptors.P1_I1_Report;// Report Desc Base Pointer
								ActualDescLength = P1_I1_REPTDESC_LEN;
							}
#endif
							else
							{
								RequestError();		// Inappropriate Request -> Interface number not valid.
								return;
							}
						}
						break;

#endif	// End of  HID_APPLICATION section.

					default : 
						RequestError();		// Inappropriate Request
						return;
					}	// End of Descriptor Type Selection
					wLength_temp = USBwLength;
					USBwLength = min(wLength_temp,ActualDescLength);	// Total Number of bytes to be sent to the host
					return;			// End of Case GET_DESCRIPTOR preprocess
#ifdef USE_USB_GET_INTERFACE
		        	case GET_INTERFACE:   		USBwLength = 1; // Return 1 byte maximum (spec)
		        					GetInterface();
        						           			break;
#endif
				case GET_CONFIGURATION :	USBwLength = 1; // Return 1 byte maximum (spec)
								UsbReport[0] = ConfigValue;
											break;

				case GET_STATUS :		USBwLength = 2; // Return 2 bytes maximum (spec)
								GetStatus(); 
											break;

				default:			RequestError();		// Inappropriate Request
											return;
				}
				break;						// End of STANDARD Requests

			case CLASS:
				switch (USBbRequest)
				{
				case GET_REPORT :			USBLibStatus |= APP_REQUEST;
											break;
#ifdef USE_HID_IDLE
				case GET_IDLE :				USBLibStatus |= APP_REQUEST;				
											break;
#endif
#ifdef USE_HID_PROTOCOL
				case GET_PROTOCOL :			USBLibStatus |= APP_REQUEST;			
											break;
#endif
#ifdef USE_DEVICE_FIRMWARE_UPGRADE
				case DFU_UPLOAD :
				case DFU_GETSTATUS :
				case DFU_GETSTATE :
											USBLibStatus |= APP_REQUEST;			
											break;
#endif
				default:					RequestError();		// Inappropriate Request
											return;
				}
				break;					// End of CLASS Requests
#ifdef USE_VENDOR_REQUESTS
			case VENDOR :					USBLibStatus |= APP_REQUEST;					
											return;
#endif
			default:						RequestError();		// Inappropriate Request
											return;
			}		// End of Request Type Selection
		}		// End of Data Stage IN Preprocess.

		else
		{// a 'Data Stage OUT' will follow the 'Setup Stage'
			USBTransferStatus = DATA_STAGE_OUT;		// Data Stage OUT will follow the Setup Stage.
													// -> Read and process data that will be received
			switch (USBbmRequestType & TYPE)
			{
			case CLASS :
				switch (USBbRequest)
				{
#ifdef USE_HID_SET_REPORT
				case SET_REPORT :			return;	// Data will be processed on next OUT Token.
#endif
#ifdef USE_DEVICE_FIRMWARE_UPGRADE
				case DFU_DNLOAD :			return;	// Data will be processed on next OUT Token.
#endif
				default :					RequestError();		// Inappropriate Request    
											return;
				}

#ifdef USE_VENDOR_REQUESTS
			case VENDOR :					return;	// Data will be processed on next OUT Token.					
#endif

			default :						RequestError();		// Inappropriate Request
			}	// end of request type selection
		}	// End of 'Data Stage Out' section
	}	// End of 'Data Stage' section	
}	// End of SetupStage()  function

/*-----------------------------------------------------------------------------
ROUTINE NAME : DataStageIn
INPUT/OUTPUT : None
DESCRIPTION  : Process 'Data Stage IN' in Control Transfer (Ep0).
-----------------------------------------------------------------------------*/ 
void DataStageIn(void)
{                   
	if (USBwLength == 8)
	{// Packet size is  8 bytes -> Need to send an empty data packet to a potential IN.
		USBTransferStatus |= ONE_MORE;
	} 
	LengthToSend = min(8,USBwLength);	// Bytes to be sent at next IN Token
	USBwLength -= LengthToSend;			// Update USBwLength (Remaining bytes to be sent)
	if (USBbRequest == GET_DESCRIPTOR)  
	{
		DataToSend = &DescAddBasePointer[USBwIndex];
		USBwIndex += LengthToSend;				            // Update Current Descriptor Pointer
	}
	else  
	{
		DataToSend = UsbReport;
	}
	Write_EP_Buffer(0,DataToSend,LengthToSend);	// Load DataToSend in EP0 DMA buff.
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : DataStageOut
INPUT/OUTPUT : None
DESCRIPTION  : Process 'Data stage OUT' in Control Transfer (Ep0).
-----------------------------------------------------------------------------*/
void DataStageOut(void)
{     
	// There is no standard request requiring a 'data stage out' 
	// -> all data stage out requests must be processed at application level
	USBLibStatus |= APP_REQUEST;
}
                                                
/*-----------------------------------------------------------------------------
ROUTINE NAME : UsbTransfer
INPUT/OUTPUT : None
DESCRIPTION  : Process a Correct Transfer Interrupt on EP0 (Enumeration process).
-----------------------------------------------------------------------------*/
void UsbTransfer(Byte TokenType)
{         
	if (TokenType & Int_Ctr_SETUP)
	{ 
		SetupStage();			// Decode the Setup Token
		if (GetEP0RxStatus() == STALL)
		{
			return;			// A request error occured in SetupStage().
		}
		else if (USBLibStatus & APP_REQUEST)
		{// This Request (CLASS or VENDOR) is handle in Handle_App_Requests()
			return;	// Will advance to STATUS OUT when returning from Handle_App_Requests()
		}
		else if (USBTransferStatus & NO_DATA_STAGE)
		{// STATUS IN expected -> Stall OUT token
			LengthToSend = 0;					// Transmit counter will be reset to 0.
			USBTransferStatus |= SET_TX0_VALID;	// Will send empty data on next IN.
			SetEP0RxStatus(STALL);              // STALL non expected OUT token.
		}
		else if (USBTransferStatus & DATA_STAGE_IN)	
		{// Initialise Data IN Transfer 
			DataStageIn();
			USBTransferStatus |= SET_RX0_VALID;			// Status OUT may occur.
			USBTransferStatus |= SET_TX0_VALID;			// Data will be sent on next IN.
		}
		else  //case USBTransferStatus = DATA_STAGE_OUT
		{// Initialise Data OUT Transfer 
			LengthToSend = 0;			// Transmit counter will be reset to 0 = Status IN
			USBTransferStatus |= SET_TX0_VALID;		// Case STATUS IN occurs.
			USBTransferStatus |= SET_RX0_VALID;		// Data will be received on next OUT Token.
		}
	} 
	else     
	if (!(USBLibStatus & APP_REQUEST))
	{// Data stage process of a Standard request
		if  (TokenType & Int_Ctr_IN)
		{// Data Stage In progress
			if (USBwLength == 0)
			{
				if (USBTransferStatus & ONE_MORE)
				{// Data length is a multiple of 8 -> Send an empty data packet on potential next IN.
					LengthToSend = 0;						// Transmit counter will be reset to 0.
					USBTransferStatus |= SET_TX0_VALID;		// Empty data will be sent on next IN.		
					USBTransferStatus |= SET_RX0_VALID;		// Status OUT may occur.
					USBTransferStatus &= ~ONE_MORE;			// Reset ONE_MORE flag
				}
				else
				{
					SetEP0TxStatus(STALL);					// All data sent -> Next IN will be stalled.
					USBTransferStatus |= SET_RX0_VALID;		// Status OUT may occur.
				}
			}
			else
			{// USBwLength != 0 -> Prepare next data transmission.   
				DataStageIn();								// Update DataToSend & LengthToSend
				USBTransferStatus |= SET_RX0_VALID;			// Status OUT may occur.
				USBTransferStatus |= SET_TX0_VALID;			// Data will be sent on next IN.
			}
		}  // end data stage IN
	   	else
		{  // Data Stage Out Progress
			DataStageOut();				// Process Received Data
		}
	}// End of standard request data stage processing
}// End of USBTransfer()


/*-----------------------------------------------------------------------------
ROUTINE NAME : RequestError
INPUT/OUTPUT : None
DESCRIPTION  : Stall next Data or Status Stage. 
				Function called when an irrelevant request has been received.
-----------------------------------------------------------------------------*/
void RequestError(void)
{	// RequestError is called only during enumeration -> STALL EP0 both dir.
	SetEP0RxStatus(STALL);
	SetEP0TxStatus(STALL);
}

#ifdef HIWARE                               
#pragma CODE_SEG DEFAULT
#endif

/*** END OF FILE ***/
