/*** (c) STMicroelectronics ***************************************************
#
# PROJECT : ST7 USB LOW SPEED LIBRARY
#
# COMPILER : COSMIC / HIWARE
#
# VERSION : 4.21
#
# DESCRIPTION : USB lib interface file
#               -->> This file contains all functions accessible by Application 
#               
# HISTORY :  08/23/00 -> Added missing "return FALSE;" , line 329 , in Test_EP_Ready()
#		 08/24/00 -> Added "SetEP0RxStatus(STALL);" line 125
#            08/24/00 -> Removed "CheckValidResume()" because now called by INT_EndSusp() ISR.
# V4.10	Init_USB_HW and Disable_USB_HW modified           
******************************************************************************/   

#include "USB_Lib.h"
#include "USB_Opts.h"
#include "Macro.h"
#include "USB_Def.h"
#include "Lib_Bits.h"
#include "USB_Var.h"
#include "USB_Rc.h"
#include "USB.h"

#ifdef HIWARE
#include MAP_FILE
#pragma CODE_SEG USBLIB_ROM
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : Init_USB_HW
INPUT/OUTPUT : None
DESCRIPTION  : Function called by user to switch on the USB cell.  
				-> Connect the peripheral to the bus
-----------------------------------------------------------------------------*/
void Init_USB_HW (void)
{
	/* Reset Usb variables used in Handle_USB_Events() */
	UsbCtrFlag = 0;				// Clear CTR Int. flags
	ConfigValue = 0;			// Default Configuration. 
	/* Initialize DMA engine and USB Cell */
	SetDmaAdd((Word) &EP0OutBuffer[0]);	// set DMA buffer starting address 
	USBIMR = 0;				// Mask all USB interrupts
	USBCTLR = 0;				// Switch the USB Voltage Regulator ON	
	USBISTR = 0;				// Clear all USB interrupts flags
	USBIMR = 0x86;				// Enable SUSP, ESUSP and RESET USB interrupts only
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : Disable_USB_HW
INPUT/OUTPUT : None
DESCRIPTION  : Function called by user to switch off the USB cell. 
				-> Disconnect the peripheral from the bus
-----------------------------------------------------------------------------*/
void Disable_USB_HW (void)
{
	USBIMR = 0;           // Mask all USB interrupts
	USBCTLR = 0x06;       // Switch the USB Voltage Regulator OFF
	USBISTR = 0;          // Clear all USB interrupts flags
	UsbCtrFlag = 0;       // Clear CTR Int. flags
	USBLibStatus &= 0x07; // Reset lib status bits. Keep product selection.
}


/*-----------------------------------------------------------------------------
ROUTINE NAME : Handle_USB_Events
INPUT/OUTPUT : None
DESCRIPTION  : Handle all USB Interrupt and process STANDARD requests.
-----------------------------------------------------------------------------*/
void Handle_USB_Events(void)
{	
	if (UsbCtrFlag != 0)  
	{	
		// One USB transfer interrupt occured -> Process it here
		UsbTransfer(UsbCtrFlag);	// Service USB CTR IT
		UsbCtrFlag = 0;				// Clear flag

		// No more CTR IT are pending ->  re-enable transmission/reception on EP0.
		if (USBTransferStatus & SET_TX0_VALID)
		{	// Request to enable transmission on EP0 -> Answer to next IN token.
			Set_EP_Ready(0,EP_IN,LengthToSend);
			USBTransferStatus &= ~SET_TX0_VALID;
		}
		if (USBTransferStatus & SET_RX0_VALID)
		{	// Request to enable reception on EP0 -> ACKnowledge next OUT token.
			Set_EP_Ready(0,EP_OUT,0);
			USBTransferStatus &= ~SET_RX0_VALID;
		}
	}
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : Enable_STATUS_Stage
INPUT/OUTPUT : None
DESCRIPTION  : Function called by application once APP_REQUESTS processing
				is done in Handle_APP_Requests()
				-> Allow ST7 to ACK or STALL incoming Status Stage
-----------------------------------------------------------------------------*/
void Enable_STATUS_Stage(void)
{
	if (USBLibStatus & USB_STALL)
	{
		RequestError();
		USBLibStatus &= ~USB_STALL;     // Clear flag
		USBLibStatus &= ~APP_REQUEST;	// Clear APP_REQUEST flag
	}
	else 
	{
		if (USBTransferStatus & NO_DATA_STAGE)
		{
			SetEP0RxStatus(STALL);		// STALL ANY EXTRA OUT  
			Set_EP_Ready(0,EP_IN,0);	// ACK STATUS IN
		}
		else if (USBTransferStatus & DATA_STAGE_IN)
		{
			if (!(USBLibStatus & APP_REQUEST))
			{// "data stage in" is over -> Advance to "Stall extra IN"
				USBwLength = 0;	// will STALL ANY EXTRA IN (cf. UsbTransfer()) 
			}			
			Set_EP_Ready(0,EP_OUT,0);	// ACK STATUS OUT
		}
		else if (USBTransferStatus & DATA_STAGE_OUT)
		{
			if (!(USBLibStatus & APP_REQUEST))
			{// data stage out is over -> No more data expected
				SetEP0RxStatus(STALL);	// STALL ANY EXTRA OUT  
			}			
			Set_EP_Ready(0,EP_IN,0);	// ACK STATUS IN
		}
	}
}

#ifdef USE_DEVICE_FIRMWARE_UPGRADE
/*-----------------------------------------------------------------------------
ROUTINE NAME : Test_EP_Ready_NP
INPUT/OUTPUT : none
DESCRIPTION  : Call Test_EP_Ready
-----------------------------------------------------------------------------*/
Bool Test_EP_Ready_NP (void)
{
  return Test_EP_Ready(GParams.SEPRParams.EndPoint, GParams.SEPRParams.Direction);
}
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : Test_EP_Ready
INPUT/OUTPUT : Endpoint#, Direction
DESCRIPTION  : Check if EP buffer is ready to be written or read
				Return TRUE if ready.
-----------------------------------------------------------------------------*/
Bool Test_EP_Ready(Byte Endpoint, Byte Direction)
{
	if (Endpoint==0)
	{
		if (Direction==EP_IN)
		{
			if (GetEP0TxStatus() == NAK)	
				return TRUE;					// Data have been sent
		}
		else
		{
			if (GetEP0RxStatus() == NAK)	
				return TRUE;					// Data have been received
		}
	}
	else if (Endpoint==1)
	{
#ifdef USE_ENDPOINT1_IN
		if (Direction==EP_IN)
		{
			if (GetEP1TxStatus() == NAK)	
				return TRUE;					// Data have been sent
		}
#endif
#ifdef USE_ENDPOINT1_OUT
		if (Direction==EP_OUT)
		{
			if (GetEP1RxStatus() == NAK)	
				return TRUE;					// Data have been received
		}
#endif
	}
	else if (Endpoint==2)
	{
#ifdef USE_ENDPOINT2_IN
		if (Direction==EP_IN)
		{
			if (GetEP2TxStatus() == NAK)	
				return TRUE;					// Data have been sent
		}
#endif
#ifdef USE_ENDPOINT2_OUT
		if (Direction==EP_OUT)
		{
			if (GetEP2RxStatus() == NAK)	
				return TRUE;					// Data have been received
		}
#endif
	}
	return FALSE;
}

#ifdef USE_DEVICE_FIRMWARE_UPGRADE
/*-----------------------------------------------------------------------------
ROUTINE NAME : Set_EP_Ready_NP
INPUT/OUTPUT : none
DESCRIPTION  : Call Set_EP_Ready
-----------------------------------------------------------------------------*/
void Set_EP_Ready_NP (void)
{
  Set_EP_Ready(GParams.SEPRParams.EndPoint, GParams.SEPRParams.Direction, GParams.SEPRParams.Length);
}
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : Set_EP_Ready
INPUT/OUTPUT : Endpoint#, EP dir, Data Payload
DESCRIPTION  : Update Transmit byte counter and Set EPTx to VALID, or Set EPRx to VALID
				REM: Length is ignored when direction=EP_OUT
-----------------------------------------------------------------------------*/
void Set_EP_Ready(Byte Endpoint, Byte Direction, Byte Length)
{
	if (Endpoint==0)
	{
		if (Direction==EP_IN)
		{
			SetEP0TxTbc(Length);                                                       
			SetEP0TxStatus(VALID);				// Enable EP0 for next IN token.
		}
		else
		{
			SetEP0RxStatus(VALID);				// Enable EP0 for next OUT token.
		}
	}
	else if (Endpoint==1)
	{
#ifdef USE_ENDPOINT1_IN
		if (Direction==EP_IN)
		{
			SetEP1TxTbc(Length);                                                       
			SetEP1TxStatus(VALID);				// Enable EP1 for next IN token.
			return;
		}
#endif
#ifdef USE_ENDPOINT1_OUT
		if (Direction==EP_OUT)
		{
			SetEP1RxStatus(VALID);				// Enable EP1 for next OUT token.
			return;
		}
#endif
	}
	else if (Endpoint==2)
	{
#ifdef USE_ENDPOINT2_IN
		if (Direction==EP_IN)
		{
			SetEP2TxTbc(Length);                                                       
			SetEP2TxStatus(VALID);				// Enable EP2 for next IN token.
			return;
		}
#endif
#ifdef USE_ENDPOINT2_OUT
		if (Direction==EP_OUT)
		{
			SetEP2RxStatus(VALID);				// Enable EP2 for next OUT token.
			return;
		}
#endif
	}
}

#ifdef USE_DEVICE_FIRMWARE_UPGRADE
/*-----------------------------------------------------------------------------
ROUTINE NAME : Write_EP_Buffer_NP
INPUT/OUTPUT : none
DESCRIPTION  : Call Write_EP_Buffer
-----------------------------------------------------------------------------*/
void Write_EP_Buffer_NP (void)
{
  Write_EP_Buffer(GParams.WEPBParams.EndPoint, GParams.WEPBParams.Buffer, GParams.WEPBParams.Length);
}
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : Write_EP_Buffer
INPUT/OUTPUT : Endpoint#, Data Source Pointer, Number of bytes to transmit
DESCRIPTION  : Write data into the USB DMA RAM Transmit buffer of selected EP
-----------------------------------------------------------------------------*/
void Write_EP_Buffer (Byte Endpoint, Byte *Data, Byte DataLength)
{ 
	Byte i;

	for (i=0; i<DataLength; i++)
	{
		if (Endpoint == 0)
		{
			EP0InBuffer[i] = *(Data + i);
		}
#ifdef USE_ENDPOINT1_IN
		else if (Endpoint == 1)
		{
			EP1InBuffer[i] = *(Data + i);
		}
#endif
#ifdef USE_ENDPOINT2_IN
		else if (Endpoint == 2)
		{
			EP2InBuffer[i] = *(Data + i);
		}
#endif
	}
}

#ifdef USE_DEVICE_FIRMWARE_UPGRADE
/*-----------------------------------------------------------------------------
ROUTINE NAME : Read_EP_Buffer_NP
INPUT/OUTPUT : none
DESCRIPTION  : Call Read_EP_Buffer
-----------------------------------------------------------------------------*/
void Read_EP_Buffer_NP (void)
{
  Read_EP_Buffer(GParams.REPBParams.EndPoint, GParams.REPBParams.Buffer);
}
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : Read_EP_Buffer
INPUT/OUTPUT : Endpoint#, Data Destination pointer
DESCRIPTION  : Read data from the USB DMA Receive buffer into *DataDestination
-----------------------------------------------------------------------------*/
void Read_EP_Buffer (Byte Endpoint, Byte *DataDestination)
{ 
	Byte i;

#ifdef USE_ENDPOINT0_OUT
	if (Endpoint == 0)
	{
		for (i=0; i<OUT_DataNumberEP0; i++)
			*(DataDestination+i) = EP0OutBuffer[i];
		return;
	}
#endif
#ifdef USE_ENDPOINT1_OUT
	if (Endpoint == 1)
	{
		for (i=0; i<OUT_DataNumberEP1; i++)
			*(DataDestination+i) = EP1OutBuffer[i];
		return;
	}
#endif
#ifdef USE_ENDPOINT2_OUT
	if (Endpoint == 2)
	{
		for (i=0; i<OUT_DataNumberEP2; i++)
			*(DataDestination+i) = EP2OutBuffer[i];
		return;
	}
#endif
}

#ifdef HIWARE
#pragma CODE_SEG DEFAULT
#endif

/*** END OF FILE ***/