//-------------------------------------------------------------------------
// File:   	    lcd.c
// Date:   	   Feb, 2006
// Author: 	   ST MCU application lab
// Description:	   LCD driver for GXM12232-1SL LCD Module of STR910-EVAL
// Revision:	   2.0
//--------------------------------------------------------------------------
//  History
//  <xxOCT2005> 1.0 - created by Jerry SHU
//  <01FEB2006> 2.0 - rewritten and improved by dfm
//--------------------------------------------------------------------------
#include "couriernew9.h"
#include "syscntr.h"
#include "lcd.h"

unsigned char rm_status; // master lcd status
unsigned char rs_status; // slave lcd status

static unsigned char dot_buffer[32]; // lcd data buffer
unsigned char ucol;    //
unsigned char ulayer;  //
unsigned char mode;    //set the positive or negtive disply here

//Global routines ================================================
void LCD_Config ( void );
void LCD_Printf(unsigned char layer, unsigned char *ptr, ...);
void LCD_Ini ( void ); //lcd initialization
void ClrScr ( void );   //clear screen
char htoa_hi(unsigned char byte);	 // converts hi nibble of unsigned byte
char htoa_lo(unsigned char byte);	// (0-F hex) to ascii

//Hard routines ================================================
// LCD pin configuration for STR910-EVAL
// control signal assignment : D0-D7 -> P8, E1 -> P9.4  E2 -> P9.5  R//W -> P9.6 DI -> P9.7
void LCD_GpioConfigDir(u32 port,u8 GpioDir_bit_postition, u8 value);
void LCDWriteBit(u32 port ,u8 GpioWrite_bit_postition, u8 value);
void LCDWriteByte(u32 port , u8 value);
u8 LCDReadByte(u32 port);
void LCD_CntlConfigOut(void);
void LCD_DataConfigIn(void);
void LCD_DataConfigOut(void);
//Utils
void BusyCheck(void);
void BusyCheckSlave(void);
void OutMasterCom( unsigned char i );
void OutSlaveCom( unsigned char i ) ;
void OutMasterDat( unsigned char i );
void OutSlaveDat( unsigned char i ) ;
void SetPage( unsigned char page0, unsigned char page1 );    //set page
void SetAddress( unsigned char address0,  unsigned char address1 );  //set column
void PutChar0 ( unsigned char ch );
void PutChar1( unsigned char ch );
void DrawBmpAsc(unsigned char x0,unsigned char layer,unsigned char width,unsigned char *bmp);
void DispOneAsc(unsigned char col,unsigned char layer,unsigned char ascii_code,unsigned char mode);
void AssignFont();


//---------------------------------------------------------------------------------------
// LCD_Config
// Function:	used to merge LCD initialization and set display start line and column address.
// Parameters:	none
// Note:	only used once just before user start using void printfLCD(unsigned char *ptr, ...).
//----------------------------------------------------------------------------------------
void LCD_Config ( void )
{
       unsigned char i;
       mode = 0;
       ulayer = 0;
       ucol = 0;
	
	SYSCNTR_Pcgr1(SYSCNTR,23, 1);//enable Port9 clock
	SYSCNTR_Prr1(SYSCNTR,23,1);  //remove Port9 reset
	SYSCNTR_Pcgr1(SYSCNTR,22, 1);//enable Port8 clock
	SYSCNTR_Prr1(SYSCNTR,22,1); //remove Port8 reset
	SYSCNTR_Gpio_Emi(SYSCNTR, 0);  //configure Port8 and Port9 as GPIO mode
	LCD_CntlConfigOut();// set all control signals as output mode
	for (i=0; i<8; i++)
        {
        // SYSCNTR_GpioType(8,i,0);  //configure Port8  output mode. 0: Push Pull 1: Open Drain
          SYSCNTR_GpioType(9,i,0);   //configure Port9  output mode. 0: Push Pull 1: Open Drain
        }
	///LCD initialization/////////
	BusyCheck();
	LCD_Ini ( );	
	ClrScr( );	
	/// set display start line and column address/////////
	OutMasterCom ( 0Xc0 );
	OutSlaveCom ( 0Xc0 );
	OutMasterCom ( 0X00 );
	OutSlaveCom ( 0X00 );
	///////////////////////////////////////
}

//---------------------------------------------------------------------------------------
// LCD_Printf
// Function:	this function is used to display character and ascii code.
// Parameters:	string pointer
// Note:	user interface to use the LCD driver.
//----------------------------------------------------------------------------------------
void LCD_Printf(unsigned char layer, unsigned char *ptr, ...)
{
	unsigned char char1,char2;
	unsigned int  char3;
	unsigned char var;

        ulayer = layer;
	#ifdef __RC51__	//use ANSI stdarg
	va_list var_ptr;
	va_start(var_ptr,0);
	#else	
	unsigned char *var_ptr=(unsigned char *)(&ptr+1);
        var_ptr = var_ptr +12;
        #endif

	//if(ulayer >3)  //if page exceed the range 0 to 3,force it to 0.
        if(ulayer >= 2)  //if page exceed the range 0 to 3,force it to 0.
	ulayer = 0;
	while (*ptr != 0)
	{
		char1 = *ptr;
		char2 = *(ptr+1);
		char3 = char1;
		char3 = (char3<<8)|char2;
		if(char1 <= 128)   // deal with asc code
		{
		     if( *ptr == '\r'){ ptr++; ucol = 0;}
		else if( *ptr == '\n'){	ptr++; ucol = 0; ulayer++;		}
		else if( *ptr == '%') {	ptr++;				
				if (*ptr == 'd')
				{
					ptr++;
			 		#ifdef __RC51__	//use ANSI stdarg
 					var = *var_ptr--;
 					var_ptr--;
	                                 #else	
					//var = *var_ptr++;
                                         var = *var_ptr;
                                        var_ptr = var_ptr +4;
					#endif
					char1 = (var & 0x0F)+'0';
					DispOneAsc(ucol,ulayer,char1,mode);
					ucol+=7;
				}
				else if (*ptr == 'x')
			 	{
			 		#ifdef __RC51__	//use ANSI stdarg
 					var = *var_ptr--;
 					var_ptr--;
	                                #else	
					//var = *var_ptr++;
                                       // var = *(var_ptr+12);
					var = *var_ptr;
                                        var_ptr = var_ptr +4;
					#endif
					char1 = htoa_hi(var);
					DispOneAsc(ucol,ulayer,char1,mode);
					ucol+=7;
					char1 = htoa_lo(var);
					DispOneAsc(ucol,ulayer,char1,mode);
					ucol+=7;
					ptr++;
				}
				else if (*ptr == 'w')
				{
					ptr++;
			 		#ifdef __RC51__	//use ANSI stdarg
 					var_ptr--;
 					var = *var_ptr;
 					#else	
					//var = *var_ptr++;
                                        var_ptr = var_ptr +1;
					var = *var_ptr;
					#endif
					char1 = htoa_hi(var);
					DispOneAsc(ucol,ulayer,char1,mode);
					ucol+=7;
					char1 = htoa_lo(var);
					DispOneAsc(ucol,ulayer,char1,mode);
					ucol+=7;
			 		#ifdef __RC51__	//use ANSI stdarg
			 		var_ptr++;
 					var = *var_ptr;
			 		var_ptr-=2;			
 					#else	
					//var = *var_ptr++;
                                        var_ptr = var_ptr -1;
					var = *var_ptr;
					var_ptr = var_ptr +4;
					#endif
					char1 = htoa_hi(var);
					DispOneAsc(ucol,ulayer,char1,mode);
					ucol+=7;
					char1 = htoa_lo(var);
					DispOneAsc(ucol,ulayer,char1,mode);
					ucol+=7;
				}
				else
				{
					char1 = *ptr;
					DispOneAsc(ucol,ulayer,char1,mode);
					ucol+=7;
					ptr++;
				}
			}
			else
			{				
			DispOneAsc(ucol,ulayer,char1,mode);
			ucol+=7;
			ptr++;    // one asc need one byte in computer system
			}
		 }
		 else
		 {
		
		 }
	}
}

///////////////////////////////////////////////////////////////////////////////
// Function:	clear both slave and master LCD.
// Parameters:	none
// Note:	none
///////////////////////////////////////////////////////////////////////////////

void ClrScr( void )
{
	unsigned char ch, page;	
	for (page= 0;page<4;page++)
	{
		SetPage( page,page );
		SetAddress( 0,0 );
		for (ch=0;ch<61;ch++) {PutChar0 (0); PutChar1 (0);}
	}
	ucol = 0;
	ulayer = 0;
}	

///////////////////////////////////////////////////////////////////////////////
void LCD_GpioConfigDir(u32 port,u8 GpioDir_bit_postition, u8 value)
{
   u8 *addr;
   u8 temp;
     if (port == 8)	 addr = (u8 *)PORT8_DIR_ADDR;
     else if (port == 9) addr = (u8 *)PORT9_DIR_ADDR;
     temp = *addr & (~(1 << GpioDir_bit_postition));
    *addr = temp |(value << GpioDir_bit_postition);
}
///////////////////////////////////////////////////////////////////////////////
void LCDWriteBit(u32 port ,u8 GpioWrite_bit_postition, u8 value)
{
  u8 *addr;
  u8 temp;

	if (port == 8)		addr = (u8 *)PORT8_MASK_ADDR;
	else if (port == 9)	addr = (u8 *)PORT9_MASK_ADDR;
	 temp = *addr & (~(1 << GpioWrite_bit_postition));
         *addr = temp |  (value << GpioWrite_bit_postition);
}
///////////////////////////////////////////////////////////////////////////////
void LCDWriteByte(u32 port , u8 value)
{
 u8 *addr;

	if (port == 8)		addr = (u8 *)PORT8_MASK_ADDR;
	else if (port == 9)	addr = (u8 *)PORT9_MASK_ADDR;
	else {}
	*addr = value ;
}
///////////////////////////////////////////////////////////////////////////////
u8 LCDReadByte(u32 port)
{
  u8 *addr;

	if (port == 8)		addr = (u8 *)PORT8_MASK_ADDR;
	else if (port == 9)	addr = (u8 *)PORT9_MASK_ADDR;
	else {}
 return *addr ;
}
///////////////////////////////////////////////////////////////////////////////
void LCD_CntlConfigOut(void) // set all control signals as output mode
{
  LCD_GpioConfigDir(9,4, 1); // set E1 as Output
  LCD_GpioConfigDir(9,5, 1); // set E2 as Output
  LCD_GpioConfigDir(9,6, 1); // set R//W as Output
  LCD_GpioConfigDir(9,7, 1); // set DI as Output
}
///////////////////////////////////////////////////////////////////////////////
void LCD_DataConfigIn(void)  //when write , set D0-D7 as input mode
{
  LCD_GpioConfigDir(8,0, 0);
  LCD_GpioConfigDir(8,1, 0);
  LCD_GpioConfigDir(8,2, 0);
  LCD_GpioConfigDir(8,3, 0);
  LCD_GpioConfigDir(8,4, 0);
  LCD_GpioConfigDir(8,5, 0);
  LCD_GpioConfigDir(8,6, 0);
  LCD_GpioConfigDir(8,7, 0);
}
///////////////////////////////////////////////////////////////////////////////
void LCD_DataConfigOut(void) // when read ,set D0-D7 as output mode
{
  LCD_GpioConfigDir(8,0, 1);
  LCD_GpioConfigDir(8,1, 1);
  LCD_GpioConfigDir(8,2, 1);
  LCD_GpioConfigDir(8,3, 1);
  LCD_GpioConfigDir(8,4, 1);
  LCD_GpioConfigDir(8,5, 1);
  LCD_GpioConfigDir(8,6, 1);
  LCD_GpioConfigDir(8,7, 1);
}


///////////////////////////////////////////////////////////////////////////////
//-----------------------------------------------------FOR LCD68------------------------------
//Function:	check whether master LCD is  busy or not .
//Parameters:none
//Note:none
/*---------------------------------------------------------------------------------------
Function:	initialize master and slave LCD.
Parameters:	none
Note:	none
----------------------------------------------------------------------------------------*/
void LCD_Ini(void)  //lcd initialization
{
	OutMasterCom (SOFTWARE_RESET );
	OutSlaveCom ( SOFTWARE_RESET );
	OutMasterCom (DISPLAY_OFF );
	OutSlaveCom ( DISPLAY_OFF );
	OutMasterCom ( DYNAMIC_DRIVE );
	OutSlaveCom ( DYNAMIC_DRIVE  );
	OutMasterCom ( DUTY_CYCLE );
	OutSlaveCom ( DUTY_CYCLE  );
	OutMasterCom ( CLOCKWISE_OUTPUT );
	OutSlaveCom ( CLOCKWISE_OUTPUT );
	OutMasterCom ( READ_MODIFY_WRITE_OFF );
	OutSlaveCom ( READ_MODIFY_WRITE_OFF );
	OutMasterCom( START_COLUMN );
	OutMasterCom (START_LINE );
	OutSlaveCom (START_COLUMN );
	OutSlaveCom ( START_LINE );
	OutMasterCom (DISPLAY_ON );
	OutSlaveCom ( DISPLAY_ON );
}

///////////////////////////////////////////////////////////////////////////////
//Function:	check whether master (slave) LCD is  busy or not .
void BusyCheck(void)			// wait until BF is cleared
{
  LCD_DataConfigIn();
  LCDWriteBit(9 ,4, 0);// E1 = 0
  LCDWriteBit(9 ,6, 1);// RW = 1
  LCDWriteBit(9 ,7, 0);// DI = 0
  LCDWriteBit(9 ,4, 1);// E1 = 1
  rm_status = LCDReadByte(8);  //read status
  LCDWriteBit(9 ,4, 0);// E1 = 0 }
  while ((rm_status & 0X80))
  {
    LCDWriteBit(9 ,4, 0);// E1 = 0
    LCDWriteBit(9 ,6, 1);// RW = 1
    LCDWriteBit(9 ,7, 0);// DI = 0
    LCDWriteBit(9 ,4, 1);// E1 = 1
    rm_status = LCDReadByte(8);  //read status
    LCDWriteBit(9 ,4, 0);// E1 = 0 }
 }
}

void BusyCheckSlave(void)			// wait until BF is cleared
{
  LCD_DataConfigIn();
  LCDWriteBit(9 ,5, 0);// E2 = 0
  LCDWriteBit(9 ,6, 1);// RW = 1
  LCDWriteBit(9 ,7, 0);// DI = 0
  LCDWriteBit(9 ,5, 1);// E2 = 1
  rs_status = LCDReadByte(8);  //read status
  LCDWriteBit(9 ,5, 0);// E2 = 0 }
  while ((rs_status & 0X80))
 {
 LCDWriteBit(9 ,5, 0);// E2 = 0
 LCDWriteBit(9 ,6, 1);// RW = 1
 LCDWriteBit(9 ,7, 0);// DI = 0
 LCDWriteBit(9 ,5, 1);// E2 = 1
 rs_status = LCDReadByte(8);  //read status
 LCDWriteBit(9 ,5, 0);// E2 = 0 }
 }
}

///////////////////////////////////////////////////////////////////////////////
// Function:	send one byte command to master (slave) LCD.
// Parameters: 	user expected command to send to master LCD.
// Note: none
//-------------------------------------------------------------------------------
void OutMasterCom( unsigned char cc )   //   write master command
{
	BusyCheck();
	LCD_DataConfigOut();
        LCDWriteBit(9 ,4, 0);// E1 = 0
	LCDWriteBit(9 ,6, 0);// RW = 0
	LCDWriteBit(9 ,7, 0);// DI = 0
	LCDWriteBit(9 ,4, 1);// E1 = 1
	LCDWriteByte(8 , cc); //write command to data bus
	LCDWriteBit(9 ,4, 0);// E1 = 0
}

void OutSlaveCom( unsigned char cc )  //write slave command
{
	BusyCheckSlave();
	LCD_DataConfigOut();
	LCDWriteBit(9 ,5, 0);// E2 = 0
	LCDWriteBit(9 ,6, 0);// RW = 0
	LCDWriteBit(9 ,7, 0);// DI = 0
	LCDWriteBit(9 ,5, 1);// E2 = 1
	LCDWriteByte(8 , cc);//write command to data bus
	LCDWriteBit(9 ,5, 0);// E2 = 0
}

///////////////////////////////////////////////////////////////////////////////
//Function:	display one byte data to master (slave) LCD.
//Parameters:	a char
//Note:	none
//--------------------------------------------------------------------------------
void OutMasterDat(unsigned char byte )  //write master data
{
	BusyCheck();
	LCD_DataConfigOut();
	LCDWriteBit(9 ,4, 0);// E1 = 0
	LCDWriteBit(9 ,6, 0);// RW = 0
	LCDWriteBit(9 ,7, 1);// DI = 1
	LCDWriteBit(9 ,4, 1);// E1 = 1
	LCDWriteByte(8 , byte);//write data to data bus
	LCDWriteBit(9 ,4, 0);// E1 = 0
}

void OutSlaveDat( unsigned char byte )  //write slave data
{
	BusyCheckSlave();
	LCD_DataConfigOut();
	LCDWriteBit(9 ,5, 0);// E2 = 0
	LCDWriteBit(9 ,6, 0);// RW = 0
	LCDWriteBit(9 ,7, 1);// DI = 1
	LCDWriteBit(9 ,5, 1);// E2 = 1
	LCDWriteByte(8 , byte);//write data to data bus
	LCDWriteBit(9 ,5, 0);// E2 = 0
}

///////////////////////////////////////////////////////////////////////////////
/*---------------------------------------------------------------------------------------
Function:
set the display page of master LCD and slave LCD simultaneously,the page range is 0 to 3,make sure the input
will not exceed this range ,otherwise it will reach a undecided result.
Parameters: page0 is the user expected display page of slave LCD and page1 is that of master LCD.
Note:	the parameter page0 and page1 is ORed with 0xB0 respectively, the reason is to meet "set page address" command
	requirement of LCD controler SED1520,Please refer to SED1520 series spec .
----------------------------------------------------------------------------------------*/
void SetPage( unsigned char page0, unsigned char page1 )    //set page
{
	OutSlaveCom ( 0XB8|page0 );
        OutMasterCom( 0XB8|page1);
}

///////////////////////////////////////////////////////////////////////////////
/*---------------------------------------------------------------------------------------
Function: set the display column of master LCD and slave LCD simultaneously,the column range is 0 to 61.
Parameters: address0 is the user expected display column of slave LCD and address1 is that of master LCD.
Note:	the parameter address0 and address1 is ANDed with 0x7f respectively, the reason is to meet "set column address" command
	requirement of LCD controler SED1520,Please refer to SED1520 series spec
----------------------------------------------------------------------------------------*/
void SetAddress( unsigned char address0,  unsigned char address1 )  //set column
{
	OutSlaveCom ( address0&0X7F );
	OutMasterCom( address1&0X7F );
}

///////////////////////////////////////////////////////////////////////////////
// Function:	display one byte data to master (slave) LCD.
// Parameters:	ch is  the user expected data to disply on master LCD.
// Note:	note 0= slave 1=master
// ----------------------------------------------------------------------------------------*/
void PutChar1( unsigned char ch )
{
	OutMasterDat (ch);
}

void PutChar0 ( unsigned char ch )
{
	OutSlaveDat ( ch );
}

///////////////////////////////////////////////////////////////////////////////////////
void DispOneAsc(unsigned char col,unsigned char layer,unsigned char ascii_code,unsigned char mode)
{
 unsigned char  ch;
 int index;
	
 for (ch=0;ch<14;ch++)
	{
	  if(mode)
	  {	if(ch%2==0)	{dot_buffer[ch] = ~TBLASCII[ascii_code*14+ch+1];}
		else		{dot_buffer[ch] = ~TBLASCII[ascii_code*14+ch-1];}
	   }
	  else
	    {
		if(ch%2==0)
		{//dot_buffer[ch] =	ascii[ascii_code*14+ch+1];
                    index = ((ascii_code*14) + ch + 1);
		    dot_buffer[ch] =	TBLASCII[index];
		}
		else
		{//dot_buffer[ch] =	ascii[ascii_code*14+ch-1];	
                 index = ((ascii_code*14) + ch - 1);
		 dot_buffer[ch] =	TBLASCII[index];
  		}
          }
	}
  DrawBmpAsc(col,layer,7,dot_buffer);  //disply the asc code after conversion
}

/*---------------------------------------------------------------------------------------
Function: this function is used to display a character.
Parameters:
        x0:	start column address.
        layer:	if layer = 0,display the first line characters,otherwise display the second line.
	width:	the number of column of a character dot matrix.
	bmp:	the pointer of the  dot matrix data.
Note:the LCD can only display two line character,so page 0 and 1 is to display the first line,page2
    and page 3 is to display the second line.
----------------------------------------------------------------------------------------*/
void DrawBmpAsc ( unsigned char x0,unsigned char layer,unsigned char width,unsigned char *bmp)
{
	unsigned char x, address;
	unsigned char *dat;
	unsigned char i=0;
	unsigned char page =0; // if layer = 0;page = 0,so it will disply the first line
	unsigned char window = 0;  //display on master LCD

        dat = bmp;
	if(layer) page=2;   //if layer >=0;then page = 2, it will disply the second line
	for(x=x0;x<x0+width;x++)
	{
          if(x>121) return;  //the column exceeded 121 will be ignored
          if(x>60) {window =1; 	address = x%61;} //display slave LCD
          else
            address=x;
            SetPage (page,page);      //disply the upper part
	    SetAddress(address,address);
	    if (window) PutChar0(dat[i]);	else PutChar1(dat[i]);
	    SetPage (page+1,page+1);  //disply the lower part
	    SetAddress(address,address);
	    if (window) PutChar0(dat[i+1]);	else PutChar1(dat[i+1]);
	    i+=2;
	}
}

void DrawBmpP0( unsigned char x0,unsigned char layer,unsigned char width,unsigned char *bmp)
{
  unsigned char x;
  for(x=x0; x<x0+width; x++) 	SetPage (0,0); 	SetAddress(x,x); PutChar1(bmp[0]);
}

void DrawBmpP1( unsigned char x0,unsigned char layer,unsigned char width,unsigned char *bmp)
{
  unsigned char x;
  for(x=x0; x<x0+width; x++)   SetPage (1,1);  SetAddress(x,x); PutChar1(bmp[0]);
}

void DrawBmpP2( unsigned char x0,unsigned char layer,unsigned char width,unsigned char *bmp)
{
  unsigned char x;
  for(x=x0; x<x0+width; x++) 	SetPage (2,2); 	SetAddress(x,x); PutChar1(bmp[0]);
}

void DrawBmpP3( unsigned char x0,unsigned char layer,unsigned char width,unsigned char *bmp)
{
  unsigned char x;
  for(x=x0; x<x0+width; x++) 	SetPage (3,3); 	SetAddress(x,x); PutChar1(bmp[0]);
}

void DrawBmpSP0( unsigned char x0,unsigned char layer,unsigned char width,unsigned char *bmp)
{
  unsigned char x;
  for(x=x0; x<x0+width; x++) 	SetPage (0,0); 	SetAddress(x,x); PutChar0(bmp[0]);
}

void DrawBmpSP1( unsigned char x0,unsigned char layer,unsigned char width,unsigned char *bmp)
{
  unsigned char x;
  for(x=x0; x<x0+width; x++)   SetPage (1,1);  SetAddress(x,x); PutChar0(bmp[0]);
}

void DrawBmpSP2( unsigned char x0,unsigned char layer,unsigned char width,unsigned char *bmp)
{
  unsigned char x;
  for(x=x0; x<x0+width; x++) 	SetPage (2,2); 	SetAddress(x,x); PutChar0(bmp[0]);
}

void DrawBmpSP3( unsigned char x0,unsigned char layer,unsigned char width,unsigned char *bmp)
{
  unsigned char x;
  for(x=x0; x<x0+width; x++) 	SetPage (3,3); 	SetAddress(x,x); PutChar0(bmp[0]);
}




/*---------------------------------------------------------------------------------------
Function:	this function is used to convert low nibble of unsigned byte.
Parameters:	unsigned byte
Note:	none
----------------------------------------------------------------------------------------*/
char htoa_lo(unsigned char byte)	// (0-F hex) to ascii
{
	byte = byte & 0x0F;	// keep lower nibble only
	if (byte <= 0x09) return(byte + 0x30);	else return (byte + 0x37);
}

/*---------------------------------------------------------------------------------------
Function:	this function is used to converts low nibble of unsigned byte.
Parameters:	byte: this unsigned byte is ready to converting.
Note:	none
----------------------------------------------------------------------------------------*/
char htoa_hi(unsigned char byte)	 // converts hi nibble of unsigned byte
{
	byte = byte & 0xF0;	// keep upper nibble only
	byte = byte >> 4;
	if (byte <= 0x09) return(byte + 0x30);	else	return (byte + 0x37);
}



//////////////////////////////////////////////////////////////////
#if 0
void AssignFont()
{
  /* ASCII 0  */
      ascii[0] = 0x1f;
      ascii[1] = 0xe0;
      ascii[2] = 0x10
      ascii[3] = 0x20;
      ascii[4] = 0x10;
      ascii[5] = 0x20;	
      ascii[6] = 0x10;	
      ascii[7] = 0x20;
      ascii[8] = 0x10;
      ascii[9] = 0x20
      ascii[10] = 0x1f;
      ascii[11] = 0xe0;
      ascii[12] = 0x00;
      ascii[13] = 0x00;

  /* ASCII 1  */
      ascii[14] = 0x1f;
      ascii[15] = 0xe0;
      ascii[16] = 0x10
      ascii[17] = 0x20;
      ascii[18] = 0x10;
      ascii[19] = 0x20;	
      ascii[20] = 0x10;	
      ascii[21] = 0x20;
      ascii[22] = 0x10;
      ascii[23] = 0x20
      ascii[24] = 0x1f;
      ascii[25] = 0xe0;
      ascii[26] = 0x00;
      ascii[27] = 0x00;

  /* ASCII 2  */
      ascii[28] = 0x1f;
      ascii[29] = 0xe0;
      ascii[30] = 0x10
      ascii[31] = 0x20;
      ascii[32] = 0x10;
      ascii[33] = 0x20;	
      ascii[34] = 0x10;	
      ascii[35] = 0x20;
      ascii[36] = 0x10;
      ascii[37] = 0x20
      ascii[38] = 0x1f;
      ascii[39] = 0xe0;
      ascii[40] = 0x00;
      ascii[41] = 0x00;

  /* ASCII 3  */
      ascii[42] = 0x1f;
      ascii[43] = 0xe0;
      ascii[44] = 0x10
      ascii[45] = 0x20;
      ascii[46] = 0x10;
      ascii[47] = 0x20;	
      ascii[48] = 0x10;	
      ascii[49] = 0x20;
      ascii[50] = 0x10;
      ascii[51] = 0x20
      ascii[52] = 0x1f;
      ascii[53] = 0xe0;
      ascii[54] = 0x00;
      ascii[55] = 0x00;

  /* ASCII 4  */	
      ascii[56] = 0x1f;
      ascii[57] = 0xe0;
      ascii[58] = 0x10
      ascii[59] = 0x20;
      ascii[60] = 0x10;
      ascii[61] = 0x20;	
      ascii[62] = 0x10;	
      ascii[63] = 0x20;
      ascii[64] = 0x10;
      ascii[65] = 0x20
      ascii[66] = 0x1f;
      ascii[67] = 0xe0;
      ascii[68] = 0x00;
      ascii[69] = 0x00;

  /* ASCII 5  */	
      ascii[70] = 0x1f;
      ascii[71] = 0xe0;
      ascii[72] = 0x10
      ascii[73] = 0x20;
      ascii[74] = 0x10;
      ascii[75] = 0x20;	
      ascii[76] = 0x10;	
      ascii[77] = 0x20;
      ascii[78] = 0x10;
      ascii[79] = 0x20
      ascii[80] = 0x1f;
      ascii[81] = 0xe0;
      ascii[82] = 0x00;
      ascii[83] = 0x00;

  /* ASCII 6  */	
      ascii[84] = 0x1f;
      ascii[85] = 0xe0;
      ascii[86] = 0x10
      ascii[87] = 0x20;
      ascii[88] = 0x10;
      ascii[89] = 0x20;	
      ascii[90] = 0x10;	
      ascii[91] = 0x20;
      ascii[92] = 0x10;
      ascii[93] = 0x20
      ascii[94] = 0x1f;
      ascii[95] = 0xe0;
      ascii[96] = 0x00;
      ascii[97] = 0x00;


  /* ASCII 7  */	
      ascii[98] = 0x1f;
      ascii[99] = 0xe0;
      ascii[100] = 0x10
      ascii[101] = 0x20;
      ascii[102] = 0x10;
      ascii[103] = 0x20;	
      ascii[104] = 0x10;	
      ascii[105] = 0x20;
      ascii[106] = 0x10;
      ascii[107] = 0x20
      ascii[108] = 0x1f;
      ascii[109] = 0xe0;
      ascii[110] = 0x00;
      ascii[111] = 0x00;
	  	
  /* ASCII 8  */	
      ascii[112] = 0x1f;
      ascii[113] = 0xe0;
      ascii[114] = 0x10
      ascii[115] = 0x20;
      ascii[116] = 0x10;
      ascii[117] = 0x20;	
      ascii[118] = 0x10;	
      ascii[119] = 0x20;
      ascii[120] = 0x10;
      ascii[121] = 0x20
      ascii[122] = 0x1f;
      ascii[123] = 0xe0;
      ascii[124] = 0x00;
      ascii[125] = 0x00;

  /* ASCII 9  */	
      ascii[112] = 0x00;
      ascii[113] = 0x00;
      ascii[114] = 0x00;
      ascii[115] = 0x00;
      ascii[116] = 0x00;
      ascii[117] = 0x00;	
      ascii[118] = 0x00;	
      ascii[119] = 0x00;
      ascii[120] = 0x00;
      ascii[121] = 0x00
      ascii[122] = 0x00;
      ascii[123] = 0x00;
      ascii[124] = 0x00;
      ascii[125] = 0x00;

  /* ASCII 10  */	
      ascii[126] = 0x00;
      ascii[127] = 0x00;
      ascii[128] = 0x00;
      ascii[129] = 0x00;
      ascii[130] = 0x00;
      ascii[131] = 0x00;	
      ascii[132] = 0x00;	
      ascii[133] = 0x00;
      ascii[134] = 0x00;
      ascii[135] = 0x00
      ascii[136] = 0x00;
      ascii[137] = 0x00;
      ascii[138] = 0x00;
      ascii[139] = 0x00;



/* ASCII 10  */ 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  /* ASCII 10  */
/* ASCII 11  */ 0x1f,0xe0,0x10,0x20,0x10,0x20,0x10,0x20,0x10,0x20,0x1f,0xe0,0x00,0x00,  /* ASCII 11  */
/* ASCII 12  */ 0x1f,0xe0,0x10,0x20,0x10,0x20,0x10,0x20,0x10,0x20,0x1f,0xe0,0x00,0x00,  /* ASCII 12  */
/* ASCII 13  */ 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  /* ASCII 13  */
/* ASCII 14  */ 0x1f,0xe0,0x10,0x20,0x10,0x20,0x10,0x20,0x10,0x20,0x1f,0xe0,0x00,0x00,  /* ASCII 14  */
/* ASCII 15  */ 0x1f,0xe0,0x10,0x20,0x10,0x20,0x10,0x20,0x10,0x20,0x1f,0xe0,0x00,0x00,  /* ASCII 15  */
/* ASCII 16  */ 0x1f,0xe0,0x10,0x20,0x10,0x20,0x10,0x20,0x10,0x20,0x1f,0xe0,0x00,0x00,  /* ASCII 16  */
/* ASCII 17  */ 0x1f,0xe0,0x10,0x20,0x10,0x20,0x10,0x20,0x10,0x20,0x1f,0xe0,0x00,0x00,  /* ASCII 17  */
/* ASCII 18  */ 0x1f,0xe0,0x10,0x20,0x10,0x20,0x10,0x20,0x10,0x20,0x1f,0xe0,0x00,0x00,  /* ASCII 18  */
/* ASCII 19  */ 0x1f,0xe0,0x10,0x20,0x10,0x20,0x10,0x20,0x10,0x20,0x1f,0xe0,0x00,0x00,  /* ASCII 19  */
/* ASCII 20  */ 0x1f,0xe0,0x10,0x20,0x10,0x20,0x10,0x20,0x10,0x20,0x1f,0xe0,0x00,0x00,  /* ASCII 20  */
/* ASCII 21  */ 0x1f,0xe0,0x10,0x20,0x10,0x20,0x10,0x20,0x10,0x20,0x1f,0xe0,0x00,0x00,  /* ASCII 21  */
/* ASCII 22  */ 0x1f,0xe0,0x10,0x20,0x10,0x20,0x10,0x20,0x10,0x20,0x1f,0xe0,0x00,0x00,  /* ASCII 22  */
/* ASCII 23  */ 0x1f,0xe0,0x10,0x20,0x10,0x20,0x10,0x20,0x10,0x20,0x1f,0xe0,0x00,0x00,  /* ASCII 23  */
/* ASCII 24  */ 0x1f,0xe0,0x10,0x20,0x10,0x20,0x10,0x20,0x10,0x20,0x1f,0xe0,0x00,0x00,  /* ASCII 24  */
/* ASCII 25  */ 0x1f,0xe0,0x10,0x20,0x10,0x20,0x10,0x20,0x10,0x20,0x1f,0xe0,0x00,0x00,  /* ASCII 25  */
/* ASCII 26  */ 0x1f,0xe0,0x10,0x20,0x10,0x20,0x10,0x20,0x10,0x20,0x1f,0xe0,0x00,0x00,  /* ASCII 26  */
/* ASCII 27  */ 0x1f,0xe0,0x10,0x20,0x10,0x20,0x10,0x20,0x10,0x20,0x1f,0xe0,0x00,0x00,  /* ASCII 27  */
/* ASCII 28  */ 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  /* ASCII 28  */
/* ASCII 29  */ 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  /* ASCII 29  */
/* ASCII 30  */ 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  /* ASCII 30  */
/* ASCII 31  */ 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  /* ASCII 31  */
/* ASCII 32  */ 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  /* ASCII 32  */
/* ASCII 33  */ 0x00,0x00,0x00,0x00,0x00,0x00,0x13,0xe0,0x00,0x00,0x00,0x00,0x00,0x00,  /* ASCII 33  */
/* ASCII 34  */ 0x00,0x00,0x00,0xe0,0x00,0x20,0x00,0x00,0x00,0xe0,0x00,0x20,0x00,0x00,  /* ASCII 34  */
/* ASCII 35  */ 0x00,0x00,0x35,0x00,0x0f,0x80,0x35,0x60,0x0f,0x80,0x05,0x60,0x00,0x00,  /* ASCII 35  */
/* ASCII 36  */ 0x00,0x00,0x0d,0x80,0x0a,0x40,0x3a,0x60,0x06,0x40,0x00,0x00,0x00,0x00,  /* ASCII 36  */
/* ASCII 37  */ 0x00,0x00,0x02,0x40,0x02,0xa0,0x0a,0x40,0x15,0x00,0x09,0x00,0x00,0x00,  /* ASCII 37  */
/* ASCII 38  */ 0x00,0x00,0x0c,0x00,0x13,0x00,0x14,0x80,0x08,0x80,0x14,0x00,0x00,0x00,  /* ASCII 38  */
/* ASCII 39  */ 0x00,0x00,0x00,0x00,0x00,0x00,0x01,0xe0,0x00,0x00,0x00,0x00,0x00,0x00,  /* ASCII 39  */
/* ASCII 40  */ 0x00,0x00,0x00,0x00,0x00,0x00,0x1f,0x80,0x60,0x60,0x00,0x00,0x00,0x00,  /* ASCII 40  */
/* ASCII 41  */ 0x00,0x00,0x00,0x00,0x60,0x60,0x1f,0x80,0x00,0x00,0x00,0x00,0x00,0x00,  /* ASCII 41  */
/* ASCII 42  */ 0x00,0x00,0x00,0x40,0x03,0x40,0x00,0xe0,0x03,0x40,0x00,0x40,0x00,0x00,  /* ASCII 42  */
/* ASCII 43  */ 0x02,0x00,0x02,0x00,0x02,0x00,0x1f,0xc0,0x02,0x00,0x02,0x00,0x02,0x00,  /* ASCII 43  */
/* ASCII 44  */ 0x00,0x00,0x00,0x00,0x60,0x00,0x38,0x00,0x08,0x00,0x00,0x00,0x00,0x00,  /* ASCII 44  */
/* ASCII 45  */ 0x00,0x00,0x02,0x00,0x02,0x00,0x02,0x00,0x02,0x00,0x02,0x00,0x00,0x00,  /* ASCII 45  */
/* ASCII 46  */ 0x00,0x00,0x00,0x00,0x18,0x00,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,  /* ASCII 46  */
/* ASCII 47  */ 0x00,0x00,0x20,0x00,0x18,0x00,0x06,0x00,0x01,0x80,0x00,0x60,0x00,0x00,  /* ASCII 47  */
/* ASCII 48  */ 0x00,0x00,0x0f,0xc0,0x10,0x20,0x10,0x20,0x10,0x20,0x0f,0xc0,0x00,0x00,  /* ASCII 48  */
/* ASCII 49  */ 0x00,0x00,0x10,0x00,0x10,0x20,0x1f,0xe0,0x10,0x00,0x10,0x00,0x00,0x00,  /* ASCII 49  */
/* ASCII 50  */ 0x00,0x00,0x18,0x40,0x14,0x20,0x12,0x20,0x11,0x20,0x18,0xc0,0x00,0x00,  /* ASCII 50  */
/* ASCII 51  */ 0x00,0x00,0x08,0x40,0x10,0x20,0x11,0x20,0x11,0x20,0x0e,0xc0,0x00,0x00,  /* ASCII 51  */
/* ASCII 52  */ 0x00,0x00,0x06,0x00,0x05,0x00,0x04,0xc0,0x14,0x20,0x1f,0xe0,0x14,0x00,  /* ASCII 52  */
/* ASCII 53  */ 0x00,0x00,0x08,0x00,0x11,0xe0,0x11,0x20,0x11,0x20,0x0e,0x20,0x00,0x00,  /* ASCII 53  */
/* ASCII 54  */ 0x00,0x00,0x0f,0x80,0x11,0x40,0x11,0x20,0x11,0x20,0x0e,0x20,0x00,0x00,  /* ASCII 54  */
/* ASCII 55  */ 0x00,0x00,0x00,0x60,0x00,0x20,0x18,0x20,0x07,0x20,0x00,0xe0,0x00,0x00,  /* ASCII 55  */
/* ASCII 56  */ 0x00,0x00,0x0e,0xc0,0x11,0x20,0x11,0x20,0x11,0x20,0x0e,0xc0,0x00,0x00,  /* ASCII 56  */
/* ASCII 57  */ 0x00,0x00,0x11,0xc0,0x12,0x20,0x12,0x20,0x0a,0x20,0x07,0xc0,0x00,0x00,  /* ASCII 57  */
/* ASCII 58  */ 0x00,0x00,0x00,0x00,0x19,0x80,0x19,0x80,0x00,0x00,0x00,0x00,0x00,0x00,  /* ASCII 58  */
/* ASCII 59  */ 0x00,0x00,0x00,0x00,0x30,0x00,0x19,0x80,0x09,0x80,0x00,0x00,0x00,0x00,  /* ASCII 59  */
/* ASCII 60  */ 0x02,0x00,0x05,0x00,0x05,0x00,0x08,0x80,0x10,0x40,0x10,0x40,0x00,0x00,  /* ASCII 60  */
/* ASCII 61  */ 0x00,0x00,0x05,0x00,0x05,0x00,0x05,0x00,0x05,0x00,0x05,0x00,0x00,0x00,  /* ASCII 61  */
/* ASCII 62  */ 0x10,0x40,0x10,0x40,0x08,0x80,0x05,0x00,0x05,0x00,0x02,0x00,0x00,0x00,  /* ASCII 62  */
/* ASCII 63  */ 0x00,0x00,0x00,0x00,0x10,0x80,0x14,0x40,0x02,0x40,0x01,0x80,0x00,0x00,  /* ASCII 63  */
/* ASCII 64  */ 0x00,0x00,0x1f,0xe0,0x20,0x10,0x23,0x10,0x24,0x90,0x17,0xe0,0x00,0x00,  /* ASCII 64  */
/* ASCII 65  */ 0x10,0x00,0x1c,0x00,0x17,0xa0,0x04,0x60,0x17,0x80,0x1c,0x00,0x10,0x00,  /* ASCII 65  */
/* ASCII 66  */ 0x10,0x20,0x1f,0xe0,0x11,0x20,0x11,0x20,0x11,0x20,0x0e,0xc0,0x00,0x00,  /* ASCII 66  */
/* ASCII 67  */ 0x00,0x00,0x0f,0xc0,0x10,0x20,0x10,0x20,0x10,0x20,0x08,0x60,0x00,0x00,  /* ASCII 67  */
/* ASCII 68  */ 0x10,0x20,0x1f,0xe0,0x10,0x20,0x10,0x20,0x08,0x40,0x07,0x80,0x00,0x00,  /* ASCII 68  */
/* ASCII 69  */ 0x10,0x20,0x1f,0xe0,0x11,0x20,0x13,0xa0,0x10,0x20,0x18,0x60,0x00,0x00,  /* ASCII 69  */
/* ASCII 70  */ 0x00,0x00,0x10,0x20,0x1f,0xe0,0x11,0x20,0x03,0xa0,0x00,0x20,0x00,0x60,  /* ASCII 70  */
/* ASCII 71  */ 0x00,0x00,0x0f,0xc0,0x10,0x20,0x10,0x20,0x12,0x20,0x0e,0x60,0x02,0x00,  /* ASCII 71  */
/* ASCII 72  */ 0x10,0x20,0x1f,0xe0,0x11,0x20,0x01,0x00,0x11,0x20,0x1f,0xe0,0x10,0x20,  /* ASCII 72  */
/* ASCII 73  */ 0x00,0x00,0x10,0x20,0x10,0x20,0x1f,0xe0,0x10,0x20,0x10,0x20,0x00,0x00,  /* ASCII 73  */
/* ASCII 74  */ 0x00,0x00,0x0e,0x00,0x10,0x20,0x10,0x20,0x0f,0xe0,0x00,0x20,0x00,0x00,  /* ASCII 74  */
/* ASCII 75  */ 0x10,0x20,0x1f,0xe0,0x12,0x20,0x03,0x00,0x04,0xa0,0x18,0x60,0x10,0x20,  /* ASCII 75  */
/* ASCII 76  */ 0x00,0x00,0x10,0x20,0x1f,0xe0,0x10,0x20,0x10,0x00,0x1c,0x00,0x00,0x00,  /* ASCII 76  */
/* ASCII 77  */ 0x10,0x20,0x1f,0xe0,0x10,0xe0,0x03,0x00,0x10,0xe0,0x1f,0xe0,0x10,0x20,  /* ASCII 77  */
/* ASCII 78  */ 0x10,0x20,0x1f,0xe0,0x10,0xe0,0x07,0x00,0x18,0x20,0x1f,0xe0,0x00,0x20,  /* ASCII 78  */
/* ASCII 79  */ 0x00,0x00,0x0f,0xc0,0x10,0x20,0x10,0x20,0x10,0x20,0x0f,0xc0,0x00,0x00,  /* ASCII 79  */
/* ASCII 80  */ 0x00,0x00,0x10,0x20,0x1f,0xe0,0x12,0x20,0x02,0x20,0x01,0xc0,0x00,0x00,  /* ASCII 80  */
/* ASCII 81  */ 0x00,0x00,0x0f,0xc0,0x10,0x20,0x30,0x20,0x30,0x20,0x2f,0xc0,0x00,0x00,  /* ASCII 81  */
/* ASCII 82  */ 0x10,0x20,0x1f,0xe0,0x12,0x20,0x02,0x20,0x06,0x20,0x09,0xc0,0x10,0x00,  /* ASCII 82  */
/* ASCII 83  */ 0x00,0x00,0x18,0xc0,0x09,0x20,0x11,0x20,0x11,0x40,0x0e,0x60,0x00,0x00,  /* ASCII 83  */
/* ASCII 84  */ 0x00,0x60,0x00,0x20,0x10,0x20,0x1f,0xe0,0x10,0x20,0x00,0x20,0x00,0x60,  /* ASCII 84  */
/* ASCII 85  */ 0x00,0x20,0x0f,0xe0,0x10,0x20,0x10,0x00,0x10,0x20,0x0f,0xe0,0x00,0x20,  /* ASCII 85  */
/* ASCII 86  */ 0x00,0x20,0x00,0xe0,0x07,0x20,0x18,0x00,0x07,0x20,0x00,0xe0,0x00,0x20,  /* ASCII 86  */
/* ASCII 87  */ 0x00,0x20,0x0f,0xe0,0x10,0x20,0x0f,0x00,0x10,0x20,0x0f,0xe0,0x00,0x20,  /* ASCII 87  */
/* ASCII 88  */ 0x10,0x20,0x18,0x60,0x04,0x80,0x03,0x00,0x04,0x80,0x18,0x60,0x10,0x20,  /* ASCII 88  */
/* ASCII 89  */ 0x00,0x20,0x00,0x60,0x11,0xa0,0x1e,0x00,0x11,0xa0,0x00,0x60,0x00,0x20,  /* ASCII 89  */
/* ASCII 90  */ 0x00,0x00,0x18,0x60,0x14,0x20,0x13,0x20,0x10,0xa0,0x18,0x60,0x00,0x00,  /* ASCII 90  */
/* ASCII 91  */ 0x00,0x00,0x00,0x00,0x7f,0xe0,0x40,0x20,0x40,0x20,0x00,0x00,0x00,0x00,  /* ASCII 91  */
/* ASCII 92  */ 0x00,0x00,0x00,0x20,0x01,0xc0,0x06,0x00,0x38,0x00,0x00,0x00,0x00,0x00,  /* ASCII 92  */
/* ASCII 93  */ 0x00,0x00,0x00,0x00,0x40,0x20,0x40,0x20,0x7f,0xe0,0x00,0x00,0x00,0x00,  /* ASCII 93  */
/* ASCII 94  */ 0x00,0x00,0x01,0x00,0x00,0x80,0x00,0x60,0x00,0x80,0x01,0x00,0x00,0x00,  /* ASCII 94  */
/* ASCII 95  */ 0x80,0x00,0x80,0x00,0x80,0x00,0x80,0x00,0x80,0x00,0x80,0x00,0x80,0x00,  /* ASCII 95  */
/* ASCII 96  */ 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x00,0x40,0x00,0x00,0x00,0x00,  /* ASCII 96  */
/* ASCII 97  */ 0x00,0x00,0x0d,0x00,0x12,0x80,0x12,0x80,0x12,0x80,0x1f,0x00,0x10,0x00,  /* ASCII 97  */
/* ASCII 98  */ 0x10,0x20,0x1f,0xe0,0x11,0x00,0x10,0x80,0x10,0x80,0x0f,0x00,0x00,0x00,  /* ASCII 98  */
/* ASCII 99  */ 0x00,0x00,0x0f,0x00,0x10,0x80,0x10,0x80,0x10,0x80,0x09,0x80,0x00,0x00,  /* ASCII 99  */
/* ASCII 100  */ 0x00,0x00,0x0f,0x00,0x10,0x80,0x10,0x80,0x11,0x20,0x1f,0xe0,0x10,0x00,  /* ASCII 100  */
/* ASCII 101  */ 0x00,0x00,0x0f,0x00,0x12,0x80,0x12,0x80,0x12,0x80,0x13,0x00,0x00,0x00,  /* ASCII 101  */
/* ASCII 102  */ 0x00,0x00,0x10,0x80,0x1f,0xc0,0x10,0xa0,0x10,0xa0,0x10,0xa0,0x00,0x00,  /* ASCII 102  */
/* ASCII 103  */ 0x00,0x00,0x0f,0x00,0x50,0x80,0x50,0x80,0x51,0x00,0x3f,0x80,0x00,0x80,  /* ASCII 103  */
/* ASCII 104  */ 0x10,0x20,0x1f,0xe0,0x11,0x00,0x00,0x80,0x10,0x80,0x1f,0x00,0x10,0x00,  /* ASCII 104  */
/* ASCII 105  */ 0x00,0x00,0x10,0x80,0x10,0x80,0x1f,0xa0,0x10,0x00,0x10,0x00,0x00,0x00,  /* ASCII 105  */
/* ASCII 106  */ 0x00,0x00,0x40,0x80,0x40,0x80,0x40,0xa0,0x3f,0x80,0x00,0x00,0x00,0x00,  /* ASCII 106  */
/* ASCII 107  */ 0x10,0x20,0x1f,0xe0,0x02,0x00,0x16,0x80,0x19,0x80,0x10,0x80,0x00,0x00,  /* ASCII 107  */
/* ASCII 108  */ 0x00,0x00,0x10,0x00,0x10,0x20,0x1f,0xe0,0x10,0x00,0x10,0x00,0x00,0x00,  /* ASCII 108  */
/* ASCII 109  */ 0x10,0x80,0x1f,0x80,0x10,0x80,0x1f,0x00,0x10,0x80,0x1f,0x00,0x10,0x00,  /* ASCII 109  */
/* ASCII 110  */ 0x10,0x80,0x1f,0x80,0x11,0x00,0x00,0x80,0x10,0x80,0x1f,0x00,0x10,0x00,  /* ASCII 110  */
/* ASCII 111  */ 0x00,0x00,0x0f,0x00,0x10,0x80,0x10,0x80,0x10,0x80,0x0f,0x00,0x00,0x00,  /* ASCII 111  */
/* ASCII 112  */ 0x40,0x80,0x7f,0x80,0x51,0x00,0x10,0x80,0x10,0x80,0x0f,0x00,0x00,0x00,  /* ASCII 112  */
/* ASCII 113  */ 0x00,0x00,0x0f,0x00,0x10,0x80,0x10,0x80,0x51,0x00,0x7f,0x80,0x40,0x80,  /* ASCII 113  */
/* ASCII 114  */ 0x00,0x00,0x10,0x80,0x1f,0x80,0x11,0x00,0x10,0x80,0x10,0x80,0x00,0x00,  /* ASCII 114  */
/* ASCII 115  */ 0x00,0x00,0x19,0x00,0x12,0x80,0x12,0x80,0x12,0x80,0x0d,0x80,0x00,0x00,  /* ASCII 115  */
/* ASCII 116  */ 0x00,0x00,0x00,0x80,0x0f,0xc0,0x10,0x80,0x10,0x80,0x10,0x80,0x08,0x00,  /* ASCII 116  */
/* ASCII 117  */ 0x00,0x80,0x0f,0x80,0x10,0x00,0x10,0x00,0x08,0x80,0x1f,0x80,0x10,0x00,  /* ASCII 117  */
/* ASCII 118  */ 0x00,0x80,0x03,0x80,0x0c,0x80,0x10,0x00,0x0c,0x80,0x03,0x80,0x00,0x80,  /* ASCII 118  */
/* ASCII 119  */ 0x00,0x80,0x0f,0x80,0x10,0x80,0x0e,0x00,0x10,0x80,0x0f,0x80,0x00,0x80,  /* ASCII 119  */
/* ASCII 120  */ 0x10,0x80,0x19,0x80,0x06,0x00,0x06,0x00,0x19,0x80,0x10,0x80,0x00,0x00,  /* ASCII 120  */
/* ASCII 121  */ 0x00,0x80,0x41,0x80,0x46,0x80,0x78,0x00,0x4c,0x80,0x03,0x80,0x00,0x80,  /* ASCII 121  */
/* ASCII 122  */ 0x00,0x00,0x19,0x80,0x14,0x80,0x12,0x80,0x11,0x80,0x18,0x80,0x00,0x00,  /* ASCII 122  */
/* ASCII 123  */ 0x00,0x00,0x00,0x00,0x04,0x00,0x3b,0xc0,0x40,0x20,0x00,0x00,0x00,0x00,  /* ASCII 123  */
/* ASCII 124  */ 0x00,0x00,0x00,0x00,0x00,0x00,0x3f,0xe0,0x00,0x00,0x00,0x00,0x00,0x00,  /* ASCII 124  */
/* ASCII 125  */ 0x00,0x00,0x00,0x00,0x40,0x20,0x3b,0xc0,0x04,0x00,0x00,0x00,0x00,0x00,  /* ASCII 125  */
/* ASCII 126  */ 0x00,0x00,0x04,0x00,0x02,0x00,0x04,0x00,0x04,0x00,0x02,0x00,0x00,0x00,  /* ASCII 126  */
};
#endif

