/*****************************************************************************
 * Copyright C 2005 ST MicroElectronics. - All rights reserved                                                                                  																			
 * Author             : Deepak Sharma				
 * Date First Issued  : 05/26/2005				
 * File Name : syscntr.c                     	
 * Purpose   : Function defintion for syscntr block related library funtions
 * History:
 *  05/26/2005 : V0.1		initial version
 *  02/01/2006 : V1.1		dfm cosmetics + Modifs for LCD
*******************************************************************************
 THE PRESENT SOFTWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS WITH
 CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME.
 AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT
 OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE CONTENT
 OF SUCH SOFTWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING INFORMATION
 CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
 ****************************************************************************/
#include "91x_type.h"
#include "91x_map.h"
#include "91x_lib.h"
#include "syscntr.h"

/******************************************************************************
*************************** SYSCNTR Utility function **************************
******************************************************************************/


/*******************************************************************************
* Function Name  : SYSCNTR_SetUsbSel
* Description    :  setup the USB_SEL selects 48MHz clockfor USB
* Input         :  	pRf            	pointer to SYSCNTR register file					
					wSettings      data (2 bits): mux selection
                                                        00-> PLL clock
                                                        01-> PLL clock/2
                                                        10-> external clock
                                                        11 -> Reserved
* Return         :  SYSCNTR_NOK on error                 	SYSCNTR_OK on right transaction
*******************************************************************************/
s32 SYSCNTR_SetUsbSel(__SYSCNTR_Reg_File *pRf,u32 wSettings)
{
	if (pRf == NULL)
		return   SYSCNTR_NOK;

	pRf->clk_cntr = (((wSettings & SYSCNTR_TWOBIT_MASK) << 10) | (pRf->clk_cntr & SYSC_CLKCNTR_USBSEL_MASK));
	return SYSCNTR_OK;

}
/*******************************************************************************
* Function Name  : SYSCNTR_SetAhbRatio
* Description    :  setup the AHB clock ratio respect to ARM CORE CLOCK clk
* Input         :  	pRf            	pointer to SYSCNTR register file					
                	wSettings      data (2 bits): mux selection
                                                        00-> hclk=clk
                                                        01-> hclk=clk/2
                                                        10-> hclk=clk/4
                                                        11-> hclk=clk/8			   					
* Return         :  SYSCNTR_NOK on error
                 	SYSCNTR_OK on right transaction
*******************************************************************************/
s32 SYSCNTR_SetAhbRatio(__SYSCNTR_Reg_File *pRf,u32 wSettings)
{
	if (pRf == NULL)
		return   SYSCNTR_NOK;

	pRf->clk_cntr = (((wSettings & SYSCNTR_TWOBIT_MASK) << 5) | (pRf->clk_cntr & SYSC_CLKCNTR_AHBDIV_MASK));
	return SYSCNTR_OK;
}

/*******************************************************************************
* Function Name  : SYSCNTR_SetApbRatio
* Description    : setup the APB clock ratio respect to ARM CORE CLOCK clk
* Input         :  	pRf            	pointer to SYSCNTR register file					
                	wSettings      data (2 bits): mux selection
                                                        00-> pclk=hclk
                                                        01-> pclk=hclk/2
                                                        10-> pclk=hclk/4
                                                        11-> pclk=hclk/8			   					
* Return         :  SYSCNTR_NOK on error
                 	SYSCNTR_OK on right transaction
*******************************************************************************/
s32 SYSCNTR_SetApbRatio(__SYSCNTR_Reg_File *pRf,u32 wSettings)
{
	if (pRf == NULL)
		return   SYSCNTR_NOK;

	pRf->clk_cntr = (((wSettings & SYSCNTR_TWOBIT_MASK) << 7) | (pRf->clk_cntr & SYSC_CLKCNTR_APBDIV_MASK));
   		return SYSCNTR_OK;

}

/*******************************************************************************
* Function Name  : SYSCNTR_SetClkRatio
* Description    : setup the CLK clock ratio respect to the main clock MCU_CLK (from pll,osc or 32KHz)
* Input         :  	pRf            	pointer to SYSCNTR register file
					wSettings       clock ratio select data:
														000-> clk=MCU_CLK
                                                        001-> clk=MCU_CLK/2
                                                        010-> clk=MCU_CLK/4
                                                        011-> clk=MCU_CLK/8
                                                        100-> clk=MCU_CLK/16
                                                        101-> clk=MCU_CLK/1024
                                                        others reserved
* Return         :  SYSCNTR_NOK on error
                 	SYSCNTR_OK on right transaction
*******************************************************************************/
u32 SYSCNTR_SetClkRatio(__SYSCNTR_Reg_File *pRf,u32 wSettings)
{
	if (pRf == NULL) return   SYSCNTR_NOK;

   pRf->clk_cntr = (((wSettings & SYSCNTR_3BIT_MASK) << 2) | (pRf->clk_cntr & SYSC_CLKCNTR_CKDIV_MASK));
   return SYSCNTR_OK;

}

/*******************************************************************************
* Function Name  : SYSCNTR_SetMcuClock
* Description    : setup the main MCU_CLK clock (from pll,osc or 32KHz)
* Input         :  	pRf            	pointer to SYSCNTR register file
					wSettings       the clock source value
														00-> PLL clock
                                                        01-> RTC clock
                                                        10-> Oscillator clock.
                                                        11-> Reserved
* Return         :  SYSCNTR_NOK on error
                 	SYSCNTR_OK on right transaction
*******************************************************************************/
s32 SYSCNTR_SetMcuClock(__SYSCNTR_Reg_File *pRf,u32 wSettings)
{
	if (pRf == NULL)		return   SYSCNTR_NOK;
	pRf->clk_cntr = ((wSettings & SYSCNTR_TWOBIT_MASK) | (pRf->clk_cntr & SYSC_CLKCNTR_CKSEL_MASK));
	return SYSCNTR_OK;
}
/*******************************************************************************
* Function Name  : SYSCNTR_DriveOutMcuClock
* Description    : Drive out 25Mhz clock
* Input         :  	pRf            	pointer to SYSCNTR register file
        	       :  SYSCNTR_NOK on error
               	SYSCNTR_OK on right transaction
******************************************************************************/
s32 SYSCNTR_DriveOutMcuClock(__SYSCNTR_Reg_File * pRf, u32 wSettings)
{
	if (pRf == NULL)		return   SYSCNTR_NOK;
	pRf->clk_cntr = ((wSettings << 12) | (pRf->clk_cntr & SYSC_MAC_SEL_MASK) );
	return SYSCNTR_OK;
}
/*******************************************************************************
* Function Name  : SYSCNTR_EnDisPll
* Description    : set/clear only the enable bit of the PLL conf register
* Input         :  	pRf            	pointer to SYSCNTR register file  	
					wSettings      data (32 bits):0-> disable, 1-> enable					
* Return         :  SYSCNTR_NOK on error
                 	SYSCNTR_OK on right transaction
*******************************************************************************/
s32 SYSCNTR_EnDisPll(__SYSCNTR_Reg_File *pRf,u32 wSettings)
{
	if (pRf == NULL) return   SYSCNTR_NOK;
	pRf->pll_conf = (((wSettings & SYSCNTR_LSB_MASK) << 19) | (pRf->pll_conf & SYSC_PLL_NOTEN_MASK));
	if(wSettings == ENABLE)
	{
		for (;;)
		if (( (STR9X_RD_WORD_REG(pRf->sys_status)) & SYSC_STATUS_LOCK_MASK ) == 1)				break;
	}	
	return SYSCNTR_OK;
}

/*******************************************************************************
* Function Name  : SYSCNTR_SetPllClk
* Description    : setup the AHB clock ratio respect to ARM CORE CLOCK clk
* Input         :  	pRf            	pointer to SYSCNTR register file
					pDiv			pDiv value for PLL calculation
					nDiv			nDiv value for PLL calculation
					mDiv			mDiv value for PLL calculation					
* Return         :  SYSCNTR_NOK on error
                 	SYSCNTR_OK on right transaction
*******************************************************************************/
s32 SYSCNTR_SetPllClk(__SYSCNTR_Reg_File *pRf,u32 pDiv, u32 nDiv, u32 mDiv)
{
	u32 tmp;

	if (pRf == NULL) return   SYSCNTR_NOK;
/*** Make sure new pDiv, nDiv, mDiv values are in the valid range*/
	if (pDiv >5)   return SYSCNTR_NOK;  //5 or 7 ??? jerry
	if (nDiv >255)	return SYSCNTR_NOK;
	if (mDiv >255)	return SYSCNTR_NOK;
	tmp = 0;
/*** Get the current value of PLL enable / disble bit*/
	tmp =((pRf->pll_conf) & SYSC_PLL_EN_MASK);
/*** Set the pll conf register with current enable/disable bit and new pDiv, nDiv, mDiv values*/
	pRf->pll_conf = tmp | (((pDiv << 16) | (nDiv << 8) | (mDiv & 0x000000FF)) & SYSC_PLL_NOTEN_MASK);
	return SYSCNTR_OK;
}

/*******************************************************************************
* Function Name  : SYSCNTR_Pcgr1
* Description    : Enable/Disable clocks to various peripheral blocks.
* Input         :  	pRf            	pointer to SYSCNTR register file
					bit_position	peripheral block (bit position in PCGR0 register)
					EnDis			nable / Disable the clock to the respective peripheral
* Return         :  SYSCNTR_NOK on error
                 	SYSCNTR_OK on right transaction
*******************************************************************************/
s32 SYSCNTR_Pcgr0(__SYSCNTR_Reg_File *pRf,u32 Pcgr0_bit_postition, u32 EnDis)
{
	u32	tmp_value;

	if (pRf == NULL)
		return   SYSCNTR_NOK;

	tmp_value = 0;	

	if((Pcgr0_bit_postition > 11) | (EnDis > 1))
		return SYSCNTR_NOK;

/* put zero in 	 Pcgr0_bit_postition in the PCGR0 register */
	tmp_value = ((pRf->pcgr0) & (~(1 << Pcgr0_bit_postition)));

/* update new value in Pcgr0_bit_postition in the PCGR0 register */
	pRf->pcgr0 = tmp_value | (EnDis << Pcgr0_bit_postition);
	
	return SYSCNTR_OK;
}

/*******************************************************************************
* Function Name  : SYSCNTR_Pcgr1
* Description    : Enable/Disable clocks to various peripheral blocks.
* Input         :  	pRf            	pointer to SYSCNTR register file
					bit_position	peripheral block (bit position in PCGR1 register)
					EnDis			nable / Disable the clock to the respective peripheral
* Return         :  SYSCNTR_NOK on error
                 	SYSCNTR_OK on right transaction
*******************************************************************************/
s32 SYSCNTR_Pcgr1(__SYSCNTR_Reg_File *pRf,u32 Pcgr1_bit_postition, u32 EnDis)
{
	u32	tmp_value;

	if (pRf == NULL)
		return   SYSCNTR_NOK;

	tmp_value = 0;	

	if((Pcgr1_bit_postition > 24) | (EnDis > 1))
		return SYSCNTR_NOK;

/* put zero in 	 Pcgr0_bit_postition in the PCGR0 register */
	tmp_value = ((pRf->pcgr1) & (~(1 << Pcgr1_bit_postition)));

/* update new value in Pcgr0_bit_postition in the PCGR0 register */
	pRf->pcgr1 = tmp_value | (EnDis << Pcgr1_bit_postition);
	
	return SYSCNTR_OK;

}

/*******************************************************************************
* Function Name  : SYSCNTR_Prr1
* Description    : Set / Clear Peripheral reset register values.
* Input         :  	pRf            	pointer to SYSCNTR register file
					bit_position	peripheral block (bit position in PRR0 register)
					Reset			Disable the Reset of the respective peripheral
* Return         :  SYSCNTR_NOK on error
                 	SYSCNTR_OK on right transaction
*******************************************************************************/
s32 SYSCNTR_Prr0(__SYSCNTR_Reg_File *pRf,u32 Prr0_bit_postition, u32 Reset)
{
	u32	tmp_value;

 	if (pRf == NULL)
		return   SYSCNTR_NOK;

	tmp_value = 0;	

	if((Prr0_bit_postition > 12) | (Reset > 1))
		return SYSCNTR_NOK;

/* put zero in 	 Pcgr0_bit_postition in the PCGR0 register */
	tmp_value = ((pRf->prr0) & (~(1 << Prr0_bit_postition)));

/* update new value in Pcgr0_bit_postition in the PCGR0 register */
	pRf->prr0 = tmp_value | (Reset << Prr0_bit_postition);
	
	return SYSCNTR_OK;

}

/*******************************************************************************
* Function Name  : SYSCNTR_Prr1
* Description    : Set / Clear Peripheral reset register values.
* Input         :  	pRf            	pointer to SYSCNTR register file
					bit_position	peripheral block (bit position in PRR1 register)
					Reset			Disable the Reset of the respective peripheral
* Return         :  SYSCNTR_NOK on error
                 	SYSCNTR_OK on right transaction
*******************************************************************************/
s32 SYSCNTR_Prr1(__SYSCNTR_Reg_File *pRf,u32 Prr1_bit_postition, u32 Reset)
{
	u32	tmp_value;

	if (pRf == NULL)
		return   SYSCNTR_NOK;

	tmp_value = 0;	

	if((Prr1_bit_postition > 24) | (Reset > 1))
		return SYSCNTR_NOK;

/* put zero in 	 Pcgr0_bit_postition in the PCGR0 register */
	tmp_value = ((pRf->prr1) & (~(1 << Prr1_bit_postition)));

/* update new value in Pcgr0_bit_postition in the PCGR0 register */
	pRf->prr1 = tmp_value | (Reset << Prr1_bit_postition);
	
	return SYSCNTR_OK;
}
/*******************************************************************************
* Function Name  : SYSCNTR_GpioOut3
* Description    : Set new GPIO register value at desired bit location
* Input         :  	pRf            	pointer to SYSCNTR register file
					bit_position	bit position in GPIO register
					value			New	value for GPIO bit
* Return         :  SYSCNTR_NOK on error      	SYSCNTR_OK on right transaction
*******************************************************************************/
s32 SYSCNTR_GpioOut3(__SYSCNTR_Reg_File *pRf,volatile u32 GpioOut3_bit_position, u32 value)
{
	u32	tmp_value;

	if (pRf == NULL)		return   SYSCNTR_NOK;
	tmp_value = 0;	
	if((GpioOut3_bit_position > 14) | (value > 3))		return SYSCNTR_NOK;
/* put zero in 	 Pcgr0_bit_position in the PCGR0 register */
	tmp_value  = pRf->gpio_out3;
	tmp_value &=  ~(3 << (GpioOut3_bit_position*2));
/* update new value in Pcgr0_bit_position in the PCGR0 register */
	pRf->gpio_out3 = tmp_value | (value << (GpioOut3_bit_position*2));
	return SYSCNTR_OK;
}
/*******************************************************************************
* Function Name  : SYSCNTR_GpioIn3
* Description    : Set new GPIO register value at desired bit location
* Input         :  	pRf            	pointer to SYSCNTR register file
			bit_position	bit position in GPIO register
			InputMode		New	input mode for GPIO bit
* Return         :  SYSCNTR_NOK on error           	SYSCNTR_OK on right transaction
*******************************************************************************/
s32 SYSCNTR_GpioIn3(__SYSCNTR_Reg_File *pRf,u32 GpioIn3_bit_postition, u32 InputMode)
{
	u32	tmp_value;

	if (pRf == NULL)
		return   SYSCNTR_NOK;

	tmp_value = 0;	

	if((GpioIn3_bit_postition > 7) | (InputMode > 1))
		return SYSCNTR_NOK;

/* put zero in 	 Pcgr0_bit_postition in the PCGR0 register */
	tmp_value = ((pRf->gpio_in3) & (~(1 << GpioIn3_bit_postition)));

/* update new value in Pcgr0_bit_postition in the PCGR0 register */
	pRf->gpio_in3 = tmp_value | (InputMode << GpioIn3_bit_postition);
	
	return SYSCNTR_OK;
}


/*******************************************************************************
* Function Name  : SYSCNTR_VicClk
* Description    : enable/disable clock VIC
* Input         :  	pRf            pointer to SYSCNTR register file
					wSettings      data (32 bits):0->disable , 1->enable		
* Return         :  SYSCNTR_NOK on error
                 	SYSCNTR_OK on right transaction
*******************************************************************************/
s32 SYSCNTR_VicClk(__SYSCNTR_Reg_File *pRf,u32 wSettings)
{
	if (pRf == NULL)		return   SYSCNTR_NOK;
	pRf->pcgr0 = (((wSettings & SYSCNTR_LSB_MASK) << 5) | (pRf->pcgr0 & SYSC_PCGR0_VIC_MASK));
	return SYSCNTR_OK;
}

/*******************************************************************************
* Function Name  : SYSCNTR_VicRst
* Description    : enable/disable reset VIC.
* Input         :  	pRf            pointer to SYSCNTR register file
			wSettings      data (32 bits):0->Reset , 1->UnReset		
* Return         :  SYSCNTR_NOK on error                 	SYSCNTR_OK on right transaction
*******************************************************************************/
s32 SYSCNTR_VicRst(__SYSCNTR_Reg_File *pRf,u32 wSettings)
{
	if (pRf == NULL)		return   SYSCNTR_NOK;
	pRf->prr0 = (((wSettings & SYSCNTR_LSB_MASK) << 5) | (pRf->prr0 & SYSC_PRR0_VIC_MASK));
	return SYSCNTR_OK;
}

///////////////////////////////////The following functions added by Jerry SHU///////////////////
/*******************************************************************************
* Function Name  : SYSCNTR_Scr0
* Description    : System configure register
* Input         :  	pRf            	pointer to SYSCNTR register file
					Scr0_bit_postition	
					value			
* Return         :  SYSCNTR_NOK on error
                 	SYSCNTR_OK on right transaction
*******************************************************************************/
s32 SYSCNTR_Scr0(__SYSCNTR_Reg_File *pRf,u32 Scr0_bit_postition, u32 value)
{
	u32	tmp_value;

	if (pRf == NULL)
		return   SYSCNTR_NOK;

	tmp_value = 0;	

	if((Scr0_bit_postition > 14) | (value > 1))
		return SYSCNTR_NOK;

/* put zero in 	 Pcgr0_bit_postition in the PCGR0 register */
	tmp_value = ((pRf->scr0) & (~(1 << Scr0_bit_postition)));

/* update new value in Pcgr0_bit_postition in the PCGR0 register */
	pRf->scr0 = tmp_value | (value << Scr0_bit_postition);
	
	return SYSCNTR_OK;
}

s32 SYSCNTR_Gpio_Emi(__SYSCNTR_Reg_File *pRf, u32 wSettings)
{
	u32	tmp_value;

	if (pRf == NULL)		return   SYSCNTR_NOK;
	tmp_value = 0;	
	if((wSettings > 1))		return SYSCNTR_NOK;
/* put zero in 	 Pcgr0_bit_postition in the PCGR0 register */
	tmp_value = ((pRf->gpio_emi) & (~1));
/* update new value in Pcgr0_bit_postition in the PCGR0 register */
	pRf->gpio_emi = tmp_value | wSettings;
	return SYSCNTR_OK;
}

/*
s32 SYSCNTR_GpioType(u32 port,u32 GpioType_bit_postition, u32 InputMode)
{
	u32	tmp_value;
	u32 * p = (u32 *)&(SYSCNTR->gpio_type0);

	tmp_value = 0;	
	if((GpioType_bit_postition > 7) | (InputMode > 1))		return SYSCNTR_NOK;
// put zero in 	 Pcgr0_bit_postition in the PCGR0 register
	tmp_value = ((p[port]) & (~(1 << GpioType_bit_postition)));
// update new value in Pcgr0_bit_postition in the PCGR0 register
	p[port] = tmp_value | (InputMode << GpioType_bit_postition);
	return SYSCNTR_OK;
}


s32 SYSCNTR_GpioType(u32 port,u32 Gpio_bit_position, u32 InputMode)
{
	u32 * p;
	u32	tmp_value;
	
	if((GpioOut_bit_position > 14) | (value > 3)) return SYSCNTR_NOK;
	if (port == 8)	  p = 0x4800e400;
	else if (port ==9)	  p = 0x4800f400;
	else ()
	  tmp_value = *p & (~(1 << GpioIn_bit_postition));
	  *p = tmp_value | (value << GpioIn_bit_postition);
	
}

*/

s32 SYSCNTR_GpioConfigOut(u32 port,u8 GpioOut_bit_postition, u8 value)
{
	u32	tmp_value;
	u32 * p = (u32 *)&(SYSCNTR->gpio_out0);

//	if (pRf == NULL)
//		return   SYSCNTR_NOK;

	tmp_value = 0;	

	if((GpioOut_bit_postition > 7) | (value > 3))
		return SYSCNTR_NOK;

/* put zero in 	 Pcgr0_bit_postition in the PCGR0 register */
	tmp_value = ((p[port]) & (~(3 << (GpioOut_bit_postition * 2)))); // ???jerry

/* update new value in Pcgr0_bit_postition in the PCGR0 register */
	p[port] = tmp_value | (value << (GpioOut_bit_postition*2));
	
	return SYSCNTR_OK;
}



s32 SYSCNTR_GpioConfigIn(u32 port,u8 GpioIn_bit_postition, u8 value)
{
	u32	tmp_value;
	u32 * p = (u32 *)&(SYSCNTR->gpio_in0);
//	if (pRf == NULL)
//		return   SYSCNTR_NOK;

	tmp_value = 0;	

	if((GpioIn_bit_postition > 7) | (value > 1))
		return SYSCNTR_NOK;

/* put zero in 	 Pcgr0_bit_postition in the PCGR0 register */
	tmp_value = ((p[port]) & (~(1 << GpioIn_bit_postition)));

/* update new value in Pcgr0_bit_postition in the PCGR0 register */
	p[port] = tmp_value | (value << GpioIn_bit_postition);
	
	return SYSCNTR_OK;
}

s32 SYSCNTR_GpioType(u32 port,u32 GpioType_bit_postition, u32 InputMode)
{
	u32	tmp_value;
	u32 * p = (u32 *)&(SYSCNTR->gpio_type0);
//	if (pRf == NULL)
//		return   SYSCNTR_NOK;

	tmp_value = 0;	

	if((GpioType_bit_postition > 7) | (InputMode > 1))
		return SYSCNTR_NOK;

/* put zero in 	 Pcgr0_bit_postition in the PCGR0 register */
	tmp_value = ((p[port]) & (~(1 << GpioType_bit_postition)));

/* update new value in Pcgr0_bit_postition in the PCGR0 register */
	p[port] = tmp_value | (InputMode << GpioType_bit_postition);
	
	return SYSCNTR_OK;
}


void GpioConfigDir(u32 port,u8 GpioDir_bit_postition, u8 value)
{
	u8 *addr;
	u8 temp;
	/*if (port == 8)
	addr = PORT8_DIR_ADDR;
	else if (port == 9)
	addr = PORT9_DIR_ADDR;*/
	
	addr = (u8 *)(0x58006400+port*0x1000);
	
	temp = *addr & (~(1 << GpioDir_bit_postition));
    *addr = temp |(value << GpioDir_bit_postition);


}

void GpioWriteBit(u32 port ,u8 GpioWrite_bit_postition, u8 value)
{
  u8 *addr;
  u8 temp;

	/*if (port == 8)
		addr = PORT8_MASK_ADDR;
		
	else if (port == 9)
		addr = PORT9_MASK_ADDR;*/
		
		addr = (u8 *)(0x580063fc+port*0x1000);
	
	 temp = *addr & (~(1 << GpioWrite_bit_postition));
	 *addr = temp |  (value << GpioWrite_bit_postition);
}

u8 GpioReadBit(u32 port ,u8 GpioWrite_bit_postition)
{
  u8 *addr;
  u8 temp;

	/*if (port == 8)
		addr = PORT8_MASK_ADDR;
		
	else if (port == 9)
		addr = PORT9_MASK_ADDR;*/
		
		addr = (u8 *)(0x580063fc+port*0x1000);
		
		temp = (*addr >> GpioWrite_bit_postition) & 0x01;
		
		return temp;
	
	// temp = *addr & (~(1 << GpioWrite_bit_postition));
	 //*addr = temp |  (value << GpioWrite_bit_postition);
}


/******************* (C) COPYRIGHT 2005 STMicroelectronics *****END OF FILE****/
